/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package autoStart;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import importOperation.ImportUtil;

public class AutoStartLoop {
	private static AutoStartLoop autoStartLoop;
	private static final String NEWLINE = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$

	private ScheduledExecutorService service;

	public static AutoStartLoop getInstance() {
		if (autoStartLoop == null) {
			synchronized (AutoStartLoop.class) {
				autoStartLoop = new AutoStartLoop();
			}
		}
		return autoStartLoop;
	}

	public void start() {
		if (service != null) {
			return;
		}
		Runnable task = new Runnable() {

			@Override
			public void run() {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						startTask();
					}
				});
			}
		};

		service = Executors.newScheduledThreadPool(6);
		service.scheduleWithFixedDelay(task, 500, 500, TimeUnit.MILLISECONDS);
	}

	public void stop() {
		if (service != null) {
			service.shutdownNow();
			service = null;
		}
	}

	void startTask() {
		String installPath = Platform.getInstallLocation().getURL().getPath().replaceFirst("/", "").replace("/", "\\"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

		// convert launches
		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		IProject[] projects = workspaceRoot.getProjects();
		try {
			convertLaunchs(projects);
		} catch (Exception e) {
			e.printStackTrace();
		}

		// listen ini files to operate
		String closeIniPath = installPath + File.separator + "close.ini"; //$NON-NLS-1$
		File closeIniFile = new File(closeIniPath);
		if (closeIniFile.exists() && closeIniFile.isFile()) {
			try {
				checkAutoCloseWorkbench(closeIniFile);
			} catch (Exception e) {
				e.printStackTrace();
				MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error", e.getMessage());
				return;
			}
		}

		String tempIniPath = installPath + File.separator + "temp.ini"; //$NON-NLS-1$
		File tempIniFile = new File(tempIniPath);
		if (tempIniFile.exists() && tempIniFile.isFile()) {
			try {
				checkAutoImportProject(tempIniFile);

				setMRSPerspective();

				String strSlnPath = null;
				try {
					strSlnPath = workspaceRoot.getPersistentProperty(new QualifiedName("workspaceAttr", "solution"));
				} catch (CoreException e) {
					e.printStackTrace();
				}

				if (strSlnPath == null) {
					strSlnPath = "";
				}

				File fileSln = new File(strSlnPath);
				if (fileSln.exists()) {
					Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
					shell.setText(TextProcessor.process(shell.getText(), TextProcessor.getDefaultDelimiters() + "-")
							+ " - " + fileSln.getAbsolutePath());
				}
			} catch (Exception e) {
				e.printStackTrace();
				MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error", e.getMessage());
				return;
			}
		}
	}

	private void convertLaunchs(IProject[] projects) throws Exception {
		for (IProject project : projects) {
			File fileLaunch = new File(project.getLocation().toOSString() + File.separator + ".launch");
			if (fileLaunch.exists()) {
				// note, if vendor is gd, 
				BufferedReader br = new BufferedReader(new FileReader(fileLaunch));
				String line = null;
				StringBuffer sb = new StringBuffer();
				boolean needConvert = false;
				while ((line = br.readLine()) != null) {
					if (line.contains("ilg.gnumcueclipse")) {
						line = line.replace("ilg.gnumcueclipse", "com.mounriver");
						needConvert = true;
					}
					sb.append(line + NEWLINE);
				}
				br.close();

				if (needConvert) {
					BufferedWriter bw = new BufferedWriter(new FileWriter(fileLaunch));
					bw.write(sb.toString());
					bw.close();

					// Add for prototype e-mrs
					File newLaunchFile = new File(fileLaunch.getParentFile().getAbsolutePath() + File.separator
							+ project.getName() + ".launch");
					if (!newLaunchFile.exists()) {
						Files.copy(fileLaunch.toPath(), newLaunchFile.toPath());
						fileLaunch.delete();
					}

				}
			}
		}
	}

	private void setMRSPerspective() {
		// Get the active page.
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				IPerspectiveRegistry reg = PlatformUI.getWorkbench().getPerspectiveRegistry();

				IPerspectiveDescriptor finalPersp = reg.findPerspectiveWithId("intro.Perspectives");
				// Set the perspective.
				page.setPerspective(finalPersp);
			}
		}
	}

	private void checkAutoCloseWorkbench(File fileCloseIni) throws Exception {
		FileReader fileReader = new FileReader(fileCloseIni);
		BufferedReader bufferedReader = new BufferedReader(fileReader);
		String line = bufferedReader.readLine();
		if (line == null) {
			fileReader.close();
			bufferedReader.close();
			return;
		}
		
		int qutIndex = line.lastIndexOf(":");
		if (qutIndex == -1) {
			fileReader.close();
			bufferedReader.close();
			return;
		}
		String strLinePath = line.substring(0, qutIndex);
		File fileSln = new File(strLinePath);

		String slnCurrentPath = ResourcesPlugin.getWorkspace().getRoot()
				.getPersistentProperty(new QualifiedName("workspaceAttr", "solution"));

		if (slnCurrentPath != null && fileSln.exists() && strLinePath.equals(slnCurrentPath)) {
			// Already opened the solution, delete close.ini & kill LoadSln.exe

			// not closed. Kill the Loadsln.exe
			Runtime.getRuntime().exec("taskkill /f /im LoadSln.exe");

			fileReader.close();
			bufferedReader.close();
			fileCloseIni.delete();
			return;
		}

		fileReader.close();
		bufferedReader.close();
		
		// always delete close.ini
		fileCloseIni.delete();

		if (!fileCloseIni.exists()) {
			// Question to close workbench
			if (PlatformUI.getWorkbench().close()) {
				return;
			} else {
				// not closed. Kill the Loadsln.exe
				Runtime.getRuntime().exec("taskkill /f /im LoadSln.exe");
			}
		}
	}

	void checkAutoImportProject(File fileTempIni) throws Exception {
		SolutionSource solution = parseTempIni(fileTempIni);
		fileTempIni.delete();

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		String slnPath = solution.getSlnPath();
		workspace.getRoot().setPersistentProperty(new QualifiedName("workspaceAttr", "solution"), slnPath);

		if (slnPath != null) {
			String str = Platform.getInstallLocation().getURL().getFile() + "ExTool" + File.separator + ".slnhistory";
			File fileSlnHistory = new File(str);
			if (!fileSlnHistory.exists()) {
				fileSlnHistory.createNewFile();
			}
			ArrayList<String> list = new ArrayList<String>();
			BufferedReader br = new BufferedReader(new FileReader(fileSlnHistory));
			String line = null;
			while ((line = br.readLine()) != null) {
				list.add(line);
			}

			if (list.size() < 5) {
				if (list.contains(slnPath)) {
					list.remove(slnPath);
					list.add(slnPath);
				} else {
					list.add(slnPath);
				}
			} else {
				if (list.contains(slnPath)) {
					list.remove(slnPath);
					list.add(slnPath);
				} else {
					list.remove(0);
					list.add(slnPath);
				}
			}

			BufferedWriter bw = new BufferedWriter(new FileWriter(fileSlnHistory));
			for (String str2 : list) {
				bw.write(str2);
				bw.write(NEWLINE);
			}

			br.close();
			bw.close();
		}

		// set workspace specific toolchain
		// arm
		InstanceScope.INSTANCE.getNode("ilg.gnumcueclipse.managedbuild.cross.arm").put("toolchain.path.962691777",
				solution.getArmPath());
		// build
		InstanceScope.INSTANCE.getNode("ilg.gnumcueclipse.managedbuild.cross").put("buildTools.path",
				solution.getBuildPath());
		// openocd
		IEclipsePreferences eprfs = InstanceScope.INSTANCE.getNode("ilg.gnumcueclipse.debug.gdbjtag.openocd");
		eprfs.put("install.folder", solution.getOpenocdPath());
		eprfs.put("executable.name", solution.getOpenocdName());

		// rv
		InstanceScope.INSTANCE.getNode("ilg.gnumcueclipse.managedbuild.cross.riscv").put("toolchain.path.512258282",
				solution.getRvPath());

		if (solution.getPrjList().isEmpty()) {
			return;
		}

		// load project to solution
		for (String projectPath : solution.getPrjList()) {
			try {
				// FORBIDDEN_CHAR = { '=', ',', '$', '#', '&', '(', ')', '{', '}' };
//				String name = new File(projectPath).getName();
//				for (char ch : FORBIDDEN_CHAR) {
//					if (name.indexOf(ch) != -1) {
//						MessageDialog.openWarning(null, Messages.AutoStartLoop_Warning,
//								NLS.bind(Messages.AutoStartLoop_WarningInfo, ch));
//						return;
//					}
//				}

				// INVALID_RESOURCE_CHARACTERS = new char[] {'\\', '/', ':', '*', '?', '"', '<',
				// '>', '|'};
//				IStatus nameStatus = IDEWorkbenchPlugin.getPluginWorkspace().validateName(name, IResource.PROJECT);
//				if (!nameStatus.isOK()) {
//					MessageDialog.openWarning(null, Messages.AutoStartLoop_Warning, nameStatus.getMessage());
//					return;
//				}

				ImportUtil.importProjectFromFolder(projectPath, new NullProgressMonitor(), slnPath);
				// IntroPlugin.closeIntro();
//						renameWVProjectFile(projectPath);
				// clean(projectPath);
			} catch (Exception e) {
				MessageDialog.openWarning(new Shell(), Messages.AutoStartLoop_Warning, e.getMessage());
			}
		}
	}

	private SolutionSource parseTempIni(File fileTemp) {
		SolutionSource solution = new SolutionSource();
		String str = ""; //$NON-NLS-1$
		BufferedReader bufferedReader = null;
		try {
			bufferedReader = new BufferedReader(new FileReader(fileTemp));
			while ((str = bufferedReader.readLine()) != null) {
				if (str.endsWith(".wvsln:")) {
					String slnPath = str.substring(0, str.length() - 1);
					solution.setSlnPath(slnPath);
					continue;
				}

				if (str.isEmpty())
					continue;
				// workspace toolchain
				if (str.startsWith("-arm:")) {
					solution.setArmPath(str.substring("-arm:".length()));
				} else if (str.startsWith("-build:")) {
					solution.setBuildPath(str.substring("-build:".length()));
				} else if (str.startsWith("-openocd name:")) {
					solution.setOpenocdName(str.substring("-openocd name:".length()));
				} else if (str.startsWith("-openocd path:")) {
					solution.setOpenocdPath(str.substring("-openocd path:".length()));
				} else if (str.startsWith("-rv:")) {
					solution.setRvPath(str.substring("-rv:".length()));
				} else {
					solution.getPrjList().add(str);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (bufferedReader != null) {
				try {
					bufferedReader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return solution;
	}
}
