/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package breakpoint;
import java.text.MessageFormat;

import org.eclipse.cdt.debug.core.CDebugCorePlugin;
import org.eclipse.cdt.debug.core.CDebugUtils;
import org.eclipse.cdt.debug.core.DebugCoreMessages;
import org.eclipse.cdt.debug.core.model.ICAddressBreakpoint;
import org.eclipse.cdt.debug.core.model.ICBreakpoint;
import org.eclipse.cdt.debug.core.model.ICBreakpointType;
import org.eclipse.cdt.debug.core.model.ICDynamicPrintf;
import org.eclipse.cdt.debug.core.model.ICFunctionBreakpoint;
import org.eclipse.cdt.debug.core.model.ICLineBreakpoint;
import org.eclipse.cdt.debug.core.model.ICWatchpoint;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.model.IBreakpoint;

public class MRSCDebugUtils  extends CDebugUtils
{
	  public static String getBreakpointText(IBreakpoint breakpoint, boolean qualified)
	    throws CoreException
	  {
	    try
	    {
	      if ((breakpoint instanceof ICAddressBreakpoint)) {
	        return getAddressBreakpointText((ICAddressBreakpoint)breakpoint, qualified);
	      }
	      if ((breakpoint instanceof ICFunctionBreakpoint)) {
	        return getFunctionBreakpointText((ICFunctionBreakpoint)breakpoint, qualified);
	      }
	      if ((breakpoint instanceof ICLineBreakpoint)) {
	        return getLineBreakpointText((ICLineBreakpoint)breakpoint, qualified);
	      }
	      if ((breakpoint instanceof ICWatchpoint)) {
	        return getWatchpointText((ICWatchpoint)breakpoint, qualified);
	      }
	      Object message = breakpoint.getMarker().getAttribute("message");
	      if (message != null) {
	        return message.toString();
	      }
	    }
	    catch (CoreException e)
	    {
	      IMarker marker = breakpoint.getMarker();
	      if ((marker == null) || (!marker.exists())) {
	        return DebugCoreMessages.getString("CDebugUtils.breakpoint_deleted");
	      }
	      CDebugCorePlugin.log(e);
	      return DebugCoreMessages.getString("CDebugUtils.exception_occurred");
	    }
	    return "";
	  }
	  
	  protected static String getLineBreakpointText(ICLineBreakpoint breakpoint, boolean qualified)
	    throws CoreException
	  {
	    StringBuffer label = new StringBuffer();
	    appendSourceName(breakpoint, label, qualified);
	    appendLineNumber(breakpoint, label);
	    if ((breakpoint instanceof ICDynamicPrintf)) {
	      appendPrintfString((ICDynamicPrintf)breakpoint, label);
	    }
	    appendBreakpointType(breakpoint, label);
	    appendExtensionMessage(breakpoint, label);
	    appendIgnoreCount(breakpoint, label);
	    appendCondition(breakpoint, label);
	    return label.toString();
	  }
	  
	  protected static StringBuffer appendBreakpointType(ICBreakpoint breakpoint, StringBuffer label)
	    throws CoreException
	  {
	    if ((breakpoint instanceof ICBreakpointType))
	    {
	      String typeString = "";
	      int type = ((ICBreakpointType)breakpoint).getType();
	      switch (type & 0xFFFFFFFE)
	      {
	      case 2: 
	        typeString = DebugCoreMessages.getString("CDebugUtils.Hardware");
	        break;
	      case 4: 
	        typeString = DebugCoreMessages.getString("CDebugUtils.Software");
	        break;
	      case 8: 
	        typeString = "Automatic";
	      }
	      if ((type & 0x1) != 0)
	      {
	        if (typeString.length() > 0) {
	          typeString = typeString + "/";
	        }
	        typeString = typeString + DebugCoreMessages.getString("CDebugUtils.Temporary");
	      }
	      if (typeString.length() > 0)
	      {
	        label.append(' ');
	        label.append(MessageFormat.format(
	          DebugCoreMessages.getString("CDebugUtils.8"), new String[] { typeString }));
	      }
	    }
	    return label;
	  }
	}
