/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package build;

import java.io.File;
import java.util.ArrayList;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.navigator.NavigatorContentService;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.INavigatorContentService;

@SuppressWarnings("restriction")
public class RebuildHandler extends AbstractBuildHandler {

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		IProject project = getSelectionProject();
		if (project != null) {
			// Add for MounRiver(Support .C file)
			ArrayList<File> fileSuffixCList = new ArrayList<File>();
			String projLocation = project.getLocation().toOSString();
			File projFile = new File(projLocation);
			addCSuffixFiles(fileSuffixCList, projFile);
			if (fileSuffixCList.size() > 0) {
				boolean result = MessageDialog.openQuestion(new Shell(),
						Messages.RebuildToolbarAction_CSuffixChangeQuestion, Messages.RebuildToolbarAction_CSuffixChangeInfo);
				if (result) {
					changeSuffixC(fileSuffixCList);
				}
			}
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
			} catch (CoreException e) {
				e.printStackTrace();
			}
			BuildUtil.rebuildProject(project.getName());
		} else {
			MessageDialog.openInformation(new Shell(), Messages.RebuildToolbarAction_0, Messages.RebuildToolbarAction_1);
		}
		return null;
	}

	private IProject getSelectionProject() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPart part = window.getActivePage().findView("org.eclipse.ui.navigator.ProjectExplorer"); //$NON-NLS-1$
			if (part != null) {
				NavigatorContentService contentService = (NavigatorContentService) Adapters.adapt(part,
						INavigatorContentService.class);
				if (contentService != null) {
					CommonViewer viewer = (CommonViewer) contentService.getViewer();
					Object selection = viewer.getSelection();
					if ((selection instanceof TreeSelection)) {
						TreeSelection treeSelection = (TreeSelection) selection;
						Object element = treeSelection.getFirstElement();
						if ((element instanceof IResource)) {
							return ((IResource) element).getProject();
						}
						if ((element instanceof ICElement)) {
							ICProject cProject = ((ICElement) element).getCProject();
							return cProject.getProject();
						}
						if ((element instanceof IEditorPart)) {
							IFile file = (IFile) ((IEditorPart) element).getEditorInput().getAdapter(IFile.class);
							return file.getProject();
						}
					}
				}
			}
		}
		return null;
	}

	/**
	 * Get .C suffix file list.
	 * 
	 * @param fileCList
	 * @param projFile
	 */
	private void addCSuffixFiles(ArrayList<File> fileCList, File projFile) {
		File[] files = projFile.listFiles();
		for (File file : files) {
			if (file.isDirectory()) {
				addCSuffixFiles(fileCList, file);
			} else {
				if (file.getName().endsWith(".C")) {
					fileCList.add(file);
				}
			}
		}
	}

	/**
	 * Change all .C suffix files to .c
	 * 
	 * @param fileSuffixCList
	 */
	private void changeSuffixC(ArrayList<File> fileSuffixCList) {
		for (File file : fileSuffixCList) {
			String newPath = file.getAbsolutePath().replace(".C", ".c");
			File tmpFile = new File(newPath);
			file.renameTo(tmpFile);
		}
	}

}