/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package convert;

import java.io.File;
import java.io.FilenameFilter;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import convertProjectUtil.XmlUtil;

public class ConvertKeilProjectDialog extends TitleAreaDialog {
	private String title;
	private final String headerTitle;
	private final String message;
	private Text textKeilPrjFile;
	private Text textTargetProjName;
	private Text textTargetPrjLoc;
	private Button btnBrowseTargetPrjLoc;
	private Button btnAutoImport;
	private Button btnGenerateComp;
	private Combo combo;

	public ConvertKeilProjectDialog(Shell parentShell) {
		this(parentShell, Messages.ConvertKeilProjectDialog_lblNewLabel_5_text,
				Messages.ConvertKeilProjectDialog_lblNewLabel_5_text,
				Messages.ConvertKeilProjectDialog_lblNewLabel_6_text); // $NON-NLS-1$
		setShellStyle(getShellStyle() | SWT.SHEET);
	}

	public ConvertKeilProjectDialog(Shell parentShell, String title, String headerTitle, String message) {
		super(parentShell);
		this.title = title;
		this.headerTitle = headerTitle;
		this.message = message;
		setShellStyle(getShellStyle());
	}

	protected void configureShell(Shell shell) {
		super.configureShell(shell);
		shell.setText(this.title);
//		shell.setSize(530, 310);
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		setTitle(this.headerTitle);
		setMessage(this.message);
		
		// top level composite
		Composite parentComposite = (Composite) super.createDialogArea(parent);
		
		Composite userComp = new Composite(parentComposite, SWT.FILL);
		GridLayout layout = new GridLayout(5, false);
		layout.marginHeight = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
		layout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
		layout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
		layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
		userComp.setLayout(layout);
		
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalIndent = 5;
		userComp.setLayoutData(gd);
		userComp.setFont(parentComposite.getFont());

		Label labelKeilPrjFile = new Label(userComp, SWT.NONE);
		labelKeilPrjFile.setText(Messages.ConvertKeilProjectDialog_6);
		gd = new GridData();
		labelKeilPrjFile.setLayoutData(gd);

		textKeilPrjFile = new Text(userComp, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		textKeilPrjFile.setLayoutData(gd);

		Button btnBrowseKeilPrjFile = new Button(userComp, SWT.NONE);
		btnBrowseKeilPrjFile.setText(Messages.ConvertKeilProjectDialog_13);
		gd = new GridData();
		btnBrowseKeilPrjFile.setLayoutData(gd);
		btnBrowseKeilPrjFile.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				FileDialog fileDialog = new FileDialog(getShell());
				fileDialog.setText(""); //$NON-NLS-1$
				fileDialog.setFilterExtensions(new String[] { "*.uvprojx;*.uvproj" }); //$NON-NLS-1$
				fileDialog.setFilterNames(new String[] { "Project Files(*.uvprojx;*.uvproj)" }); //$NON-NLS-1$
				String path = fileDialog.open();
				if (path != null) {
					textKeilPrjFile.setText(path);
					textKeilPrjFile.setToolTipText(path);
					try {
						textTargetProjName.setText(getNewProjectName(XmlUtil.getInstance().getKeilProjectName(path)));
					} catch (Exception e1) {
						e1.printStackTrace();
					}
				}
			}
		});

		Label labelTargetPrjName = new Label(userComp, SWT.NONE);
		labelTargetPrjName.setText(Messages.ConvertKeilProjectDialog_19);
		gd = new GridData();
		labelTargetPrjName.setLayoutData(gd);

		textTargetProjName = new Text(userComp, SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		textTargetProjName.setLayoutData(gd);

		Label labelTargetPrjType = new Label(userComp, SWT.NONE);
		labelTargetPrjType.setText(Messages.ConvertKeilProjectDialog_5);
		gd = new GridData();
		labelTargetPrjType.setLayoutData(gd);

		combo = new Combo(userComp, SWT.READ_ONLY);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		combo.setLayoutData(gd);

		Label labelEmpty = new Label(userComp, SWT.NONE);
		gd = new GridData();
		labelEmpty.setLayoutData(gd);

		Label labelDefaultPos = new Label(userComp, SWT.NONE);
		labelDefaultPos.setText(Messages.ConvertKeilProjectDialog_7);

		textTargetPrjLoc = new Text(userComp, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		textTargetPrjLoc.setLayoutData(gd);

		btnBrowseTargetPrjLoc = new Button(userComp, SWT.NONE);
		btnBrowseTargetPrjLoc.setText(Messages.ConvertKeilProjectDialog_16);
		gd = new GridData();
		btnBrowseTargetPrjLoc.setLayoutData(gd);
		btnBrowseTargetPrjLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				DirectoryDialog directoryDialog = new DirectoryDialog(getShell());
				directoryDialog.setText(""); //$NON-NLS-1$
				directoryDialog.setMessage(""); //$NON-NLS-1$
				String path = directoryDialog.open();
				if (path != null) {
					textTargetPrjLoc.setText(path);
					textTargetPrjLoc.setToolTipText(path);
				}
			}
		});

		Composite compBottom = new Composite(userComp, SWT.NONE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 5;
		gd.verticalIndent = 10;
		compBottom.setLayoutData(gd);
		compBottom.setLayout(new FillLayout());

		btnAutoImport = new Button(compBottom, SWT.CHECK);
		btnAutoImport.setText(Messages.ConvertKeilProjectDialog_17);
		btnAutoImport.setSelection(true);

		btnGenerateComp = new Button(compBottom, SWT.CHECK);
		btnGenerateComp.setText(Messages.ConvertKeilProjectDialog_btnCheckButton_2_text);
		btnGenerateComp.setSelection(false);

		initWidget();
		
		setHelpAvailable(false);
		Dialog.applyDialogFont(parentComposite);
		return userComp;
	}

	private void initWidget() {
		combo.setItems(getTemplateList());
		combo.select(0);
		textTargetPrjLoc.setText(getDefaultLocation());
		textTargetPrjLoc.setToolTipText(getDefaultLocation());
	}

	public String[] getTemplateList() {
		String templatePath = Platform.getInstallLocation().getURL().getFile() + "ProLib"; //$NON-NLS-1$
		File file = new File(templatePath);
		if (!file.exists() || !file.isDirectory()) {
			return new String[0];
		}
		File[] files = file.listFiles(new FilenameFilter() {

			@Override
			public boolean accept(File arg0, String arg1) {
				return arg0.isDirectory();
			}
		});
		if (files == null || files.length == 0) {
			return new String[0];
		}
		String[] retStrings = new String[files.length];
		for (int i = 0; i < retStrings.length; i++) {
			retStrings[i] = files[i].getName();
		}
		return retStrings;
	}

	private String getNewProjectName(String oldName) {
		if (oldName == null || "".equalsIgnoreCase(oldName)) { //$NON-NLS-1$
			return ""; //$NON-NLS-1$
		}
		return oldName.replaceAll("[c | C][h | H][3][2][f | F][1][0]", "CH32V10"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	private String getDefaultLocation() {
		// get target location
		String s = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
		return s.replace("/", "\\"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	@Override
	protected void okPressed() {
		if (checkValid()) {
			processConvert();
			super.okPressed();
		}
	}

	boolean checkValid() {
		if (textKeilPrjFile.getText() == null || textKeilPrjFile.getText().equalsIgnoreCase("")) { //$NON-NLS-1$
			showMessageBox(Messages.ConvertKeilProjectDialog_27);
			return false;
		}
		if (textTargetProjName.getText() == null || textTargetProjName.getText().equalsIgnoreCase("")) { //$NON-NLS-1$
			showMessageBox(Messages.ConvertKeilProjectDialog_25);
			return false;
		}
		if (combo.getText() == null || combo.getText().equalsIgnoreCase("")) { //$NON-NLS-1$
			showMessageBox(Messages.ConvertKeilProjectDialog_23);
			return false;
		}
		if (textTargetPrjLoc.getText() == null || textTargetPrjLoc.getText().equalsIgnoreCase("")) { //$NON-NLS-1$
			showMessageBox(Messages.ConvertKeilProjectDialog_29);
			return false;
		}
		String path = textTargetPrjLoc.getText() + File.separator + textTargetProjName.getText();
		File file2 = new File(path);
		if (file2.exists()) {
			showMessageBox(path + " " + Messages.ConvertKeilProjectDialog_1); //$NON-NLS-1$
			return false;
		}
		return true;
	}

	private void processConvert() {
		String path_eclipse = textTargetPrjLoc.getText();
		String name_eclipse = textTargetProjName.getText();
		boolean generateCompare = btnGenerateComp.getSelection();
		String project_eclipse_path = path_eclipse + File.separator + name_eclipse;
		String project_compare_path = project_eclipse_path + ConvertProjectImp.suffix_compare_folder;
		try {
			ConvertProjectImp.ConvertKeilProject(combo.getText(), textKeilPrjFile.getText(), name_eclipse, path_eclipse,
					btnAutoImport.getSelection(), generateCompare);
			if (generateCompare && ConvertProjectImp.RESULT) {
				MessageDialog.openInformation(getShell(), Messages.ConvertKeilProjectDialog_32,
						Messages.ConvertKeilProjectDialog_33 + "\r\n" + project_eclipse_path + "\r\n" // $NON-NLS-3$ //$NON-NLS-1$
								+ Messages.ConvertKeilProjectDialog_35 + "\r\n" + project_compare_path);
			}
		} catch (Exception e) {
			e.printStackTrace();
			showMessageBox(e.getMessage());
		}
		if (!ConvertProjectImp.RESULT) {
			deleteProject(project_eclipse_path);
			deleteProject(project_compare_path);
		}
		if (ConvertProjectImp.SHOWLOG) {
			if (MessageDialog.openQuestion(getShell(), Messages.ConvertProjectImp_42,
					Messages.ConvertKeilProjectDialog_3)) {
				File logFile = new File(ConvertProjectImp.logFilePath);

				if (!logFile.exists() || !logFile.isFile()) {
					MessageDialog.openWarning(getShell(), Messages.ConvertProjectImp_42,
							Messages.ConvertKeilProjectDialog_4);
					return;
				}
				IFileStore fileStore = EFS.getLocalFileSystem().getStore(logFile.toURI());
				IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
				try {
					IDE.openEditorOnFileStore(page, fileStore);
				} catch (PartInitException e) {
				}
			}
		}
	}

	private void deleteProject(String path) {
		File file = new File(path);
		if (!file.exists() || !file.isDirectory()) {
			return;
		}
		deleteFile(file);
	}

	private void deleteFile(File file) {
		if (!file.exists()) {
			return;
		}
		if (file.isFile() || file.list() == null) {
			file.delete();
		} else {
			File[] files = file.listFiles();
			for (File a : files) {
				deleteFile(a);
			}
			file.delete();
		}
	}

	private void showMessageBox(String msg) {
		Display display = Display.getDefault();
		if (!display.isDisposed()) {
			Runnable runnable = new Runnable() {
				public void run() {
					// Ĵ
					MessageDialog.openWarning(getShell(), Messages.ConvertKeilProjectDialog_31, msg);
				}
			};
			display.syncExec(runnable);// ؼһϣͬãȴ̴֮߳
		}
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, Messages.ConvertKeilProjectDialog_9, false);
		createButton(parent, IDialogConstants.CANCEL_ID, Messages.ConvertKeilProjectDialog_8, false);
	}
}
