/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package handlers;

import java.io.File;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.AbstractOperation;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.undo.CreateFolderOperation;
import org.eclipse.ui.ide.undo.WorkspaceUndoUtil;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.navigator.NavigatorContentService;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.INavigatorContentService;
import org.eclipse.ui.navigator.resources.ProjectExplorer;

@SuppressWarnings("restriction")
public class AddExternalLinkFoldHandler2 extends AbstractHandler {

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		Shell shell = window.getShell();
		DirectoryDialog dirDialog = new DirectoryDialog(shell, SWT.SINGLE | SWT.SHEET);
		dirDialog.setText("Add Linked Folder");
		String path = dirDialog.open();
		if (path != null) {
			File source = new File(path);
			IContainer container = getTargetResource();
			IFolder ifolder = IDEWorkbenchPlugin.getPluginWorkspace().getRoot().getFolder(
					container.getFullPath().append(source.getName()));
			IRunnableWithProgress op = monitor -> {
				AbstractOperation op1;
				op1 = new CreateFolderOperation(
					ifolder, URIUtil.toURI(source.getAbsolutePath()), false, null,
					IDEWorkbenchMessages.WizardNewFolderCreationPage_title);
				try {
					// see bug https://bugs.eclipse.org/bugs/show_bug.cgi?id=219901
					// directly execute the operation so that the undo state is
					// not preserved.  Making this undoable can result in accidental
					// folder (and file) deletions.
					op1.execute(monitor, WorkspaceUndoUtil
						.getUIInfoAdapter(shell));
				} catch (final ExecutionException e) {
					shell.getDisplay().syncExec(
							() -> {
								if (e.getCause() instanceof CoreException) {
									ErrorDialog
											.openError(
													shell, // Was Utilities.getFocusShell()
													IDEWorkbenchMessages.WizardNewFolderCreationPage_errorTitle,
													null, // no special message
													((CoreException) e
															.getCause())
															.getStatus());
								} else {
									IDEWorkbenchPlugin
											.log(
													getClass(),
													"createNewFolder()", e.getCause()); //$NON-NLS-1$
									MessageDialog
											.openError(
													shell,
													IDEWorkbenchMessages.WizardNewFolderCreationPage_internalErrorTitle,
													NLS
															.bind(
																	IDEWorkbenchMessages.WizardNewFolder_internalError,
																	e
																			.getCause()
																			.getMessage()));
								}
							});
				}
			};

			try {
				ModalContext.run(op, true, new NullProgressMonitor(), shell
						.getDisplay());
			} catch (InterruptedException e) {
				return null;
			} catch (InvocationTargetException e) {
				// ExecutionExceptions are handled above, but unexpected runtime
				// exceptions and errors may still occur.
				IDEWorkbenchPlugin.log(getClass(),
						"createNewFolder()", e.getTargetException()); //$NON-NLS-1$
				MessageDialog
						.open(MessageDialog.ERROR,
								shell,
								IDEWorkbenchMessages.WizardNewFolderCreationPage_internalErrorTitle,
								NLS
										.bind(
												IDEWorkbenchMessages.WizardNewFolder_internalError,
												e.getTargetException().getMessage()), SWT.SHEET);
				return null;
			}
			
		}
		return null;
	}

	private IContainer getTargetResource() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPart part = window.getActivePage().findView(ProjectExplorer.VIEW_ID);
			if (part != null) {
				NavigatorContentService contentService = (NavigatorContentService) Adapters.adapt(part,
						INavigatorContentService.class);
				if (contentService != null) {
					CommonViewer viewer = (CommonViewer) contentService.getViewer();
					Object selection = viewer.getSelection();
					if (selection instanceof TreeSelection) {
						TreeSelection treeSelection = (TreeSelection) selection;
						Object element = treeSelection.getFirstElement();
						if (element instanceof ICElement) {
							IResource resource = ((ICElement) element).getResource();
							if (resource instanceof IContainer) {
								return (IContainer) resource;
							}
						} else if (element instanceof IProject) {
							return (IContainer) element;
						}
					}
				}
			}
		}
		return null;
	}
}
