/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package handlers;

import java.io.File;
import java.nio.file.Files;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.navigator.NavigatorContentService;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.INavigatorContentService;
import org.eclipse.ui.navigator.resources.ProjectExplorer;

@SuppressWarnings("restriction")
public class AddFileHandler extends AbstractHandler {

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		Shell shell = window.getShell();
		FileDialog fileDialog = new FileDialog(shell, SWT.OPEN | SWT.SINGLE);
		fileDialog.setText(Messages.AddFilesHandler_0);
		fileDialog.setFilterNames(
				new String[] { Messages.AddFilesHandler_1, Messages.AddFilesHandler_2, Messages.AddFilesHandler_3 }); // $NON-NLS-1$
																														// //$NON-NLS-2$
																														// //$NON-NLS-3$
		fileDialog.setFilterExtensions(new String[] { "*.c;*.h;*.S", "*.ld;*.lds", "*.*" }); //$NON-NLS-1$//$NON-NLS-2$
		String path = fileDialog.open();
		if (path != null) {
			File source = new File(path);
			IContainer folder = getTargetResource();
			String strDesParent = folder.getLocation().toOSString();
			File fileDesParent = new File(strDesParent);
			if (fileDesParent.isDirectory()) {
				try {
					File newFile = new File(fileDesParent.getAbsolutePath() + File.separator + source.getName());
					if (newFile.exists()) {
						if (newFile.equals(source)) {
							folder.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
							gainFocus(folder, newFile);
							return null;
						} else {
							if (MessageDialog.openConfirm(shell, Messages.AddFilesHandler_5,
									source.getName() + Messages.AddFilesHandler_6)) {
								newFile.delete();
							} else {
								return null;
							}
						}
					}
					Files.copy(source.toPath(), newFile.toPath());
					folder.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
					gainFocus(folder, newFile);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		return null;
	}

	private void gainFocus(IContainer folder, File newFile) {
		Display.getDefault().asyncExec(new Runnable() {

			@Override
			public void run() {
				IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				if (window != null) {
					IWorkbenchPart part = window.getActivePage().findView(ProjectExplorer.VIEW_ID);
					if (part != null) {
						NavigatorContentService contentService = (NavigatorContentService) Adapters.adapt(part,
								INavigatorContentService.class);
						if (contentService != null) {
							CommonViewer viewer = (CommonViewer) contentService.getViewer();
							IResource[] pp = new IResource[] { folder.findMember(newFile.getName()) };
							viewer.setSelection(new TreeSelection(new TreePath(pp)));
							viewer.refresh();
						}
					}
				}
			}
		});
	}

	private IContainer getTargetResource() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPart part = window.getActivePage().findView(ProjectExplorer.VIEW_ID);
			if (part != null) {
				NavigatorContentService contentService = (NavigatorContentService) Adapters.adapt(part,
						INavigatorContentService.class);
				if (contentService != null) {
					CommonViewer viewer = (CommonViewer) contentService.getViewer();
					Object selection = viewer.getSelection();
					if (selection instanceof TreeSelection) {
						TreeSelection treeSelection = (TreeSelection) selection;
						Object element = treeSelection.getFirstElement();
						if (element instanceof ICElement) {
							IResource resource = ((ICElement) element).getResource();
							if (resource instanceof IContainer) {
								return (IContainer) resource;
							}
						} else if (element instanceof IProject) {
							return (IContainer) element;
						}
					}
				}
			}
		}
		return null;
	}
}
