/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package importOperation;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.file.Files;
import java.util.Enumeration;
import java.util.Random;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.IOverwriteQuery;
import org.eclipse.ui.internal.navigator.NavigatorContentService;
import org.eclipse.ui.internal.navigator.resources.plugin.WorkbenchNavigatorPlugin;
import org.eclipse.ui.internal.wizards.datatransfer.ZipLeveledStructureProvider;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.INavigatorContentService;
import org.eclipse.ui.navigator.resources.ProjectExplorer;
import org.eclipse.ui.wizards.datatransfer.FileSystemStructureProvider;
import org.eclipse.ui.wizards.datatransfer.ImportOperation;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import autoStart.AutoStartAction;
import convertProjectUtil.XmlMofidyUtil;

@SuppressWarnings({ "restriction", "deprecation" })
public class ImportUtil {
	private static final String NEWLINE = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
	private static final char[] FORBIDDEN_CHAR = { '=', ',', '#', '*', '&', '(', ')', '{', '}' };

	public static IProject importProjectFromZip(boolean bCreateDir, String vendor, String series, String newProjectName,
			String newProjectLocation, String link, String zipTemplatePath, IProgressMonitor monitor) throws Exception {
		File file = new File(zipTemplatePath);

		if (!file.exists()) {
			throw new Exception(Messages.ImportUtil_5);
		}

		// Ŀ¼
		IWorkspace workspace = ResourcesPlugin.getWorkspace();

		IProjectDescription newProjectDescription = workspace.newProjectDescription(newProjectName);
		IProject newProject = workspace.getRoot().getProject(newProjectName);
		URI uri = null;

		if (newProjectLocation != null) {
			if (bCreateDir) {
				uri = URIUtil.toURI(newProjectLocation + File.separator + newProjectName);
			} else {
				uri = URIUtil.toURI(newProjectLocation);
			}
		}
		// Add for MounRiver v1.5, virtual mapping
		else {
			uri = URIUtil.toURI(ResourcesPlugin.getWorkspace().getRoot().getLocation().toString() + File.separator
					+ newProjectName);
		}
		newProjectDescription.setLocationURI(uri);

		newProject.create(newProjectDescription, null);

		monitor.worked(1);
		newProject.open(null);
		monitor.worked(1);

		IOverwriteQuery overwriteQuery = new IOverwriteQuery() {
			public String queryOverwrite(String file) {
				return ALL;
			}
		};

		ZipFile zipFile = new ZipFile(zipTemplatePath);
		ZipLeveledStructureProvider provider = new ZipLeveledStructureProvider(zipFile);
		ImportOperation importOperation = new ImportOperation(newProject.getFullPath(), provider.getRoot(), provider,
				overwriteQuery);
		importOperation.setCreateContainerStructure(false);
		importOperation.run(new SubProgressMonitor(monitor, 5));
		zipFile.close();

//		syncCreateProject(newProject, vendor, series, link, newProjectName, zipTemplatePath);

		boolean isSame = modifyTmpInfoFile(newProject, vendor, series, link, zipTemplatePath);
		createWvprojFile(newProject);
		XmlMofidyUtil.getInstance().onlyModifyProjectName(newProject, newProjectName);
		XmlMofidyUtil.getInstance().modifyLaunchContent(newProject,
				newProject.getLocation().toOSString() + File.separator + ".launch");

		// if link change
		if (!isSame) {
			AutoStartAction.updateLaunchContent(newProject, vendor, link);
		}

		monitor.worked(1);
		newProject.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
		setProjActive(newProject);
		monitor.worked(1);
		return newProject;
	}

	public static void importProjectFromFolder(String projLoc, IProgressMonitor monitor, String solutionPath)
			throws Exception {
		if (projLoc == null || projLoc.isEmpty()) { // $NON-NLS-1$
			throw new Exception(Messages.ImportUtil_11);
		}
		// ѰԴĿ¼

		File file = new File(projLoc);
		if (!file.exists() || !file.isDirectory()) {
			throw new Exception(NLS.bind(Messages.ImportUtil_14, projLoc));
		}

		// Ƿ.projectļ
		File[] files = file.listFiles(new FilenameFilter() {

			@Override
			public boolean accept(File dir, String name) {
				return name.equalsIgnoreCase(IProjectDescription.DESCRIPTION_FILE_NAME); // $NON-NLS-1$
			}
		});

		File[] files2 = file.listFiles(new FilenameFilter() {

			@Override
			public boolean accept(File dir, String name) {
				return name.endsWith(".wvproj"); // $NON-NLS-1$
			}
		});

		File[] files3 = file.listFiles(new FilenameFilter() {

			@Override
			public boolean accept(File dir, String name) {
				return name.endsWith(".template"); // $NON-NLS-1$
			}
		});

		File[] files4 = file.listFiles(new FilenameFilter() {

			@Override
			public boolean accept(File dir, String name) {
				return name.endsWith(".launch"); // $NON-NLS-1$
			}
		});

		if (files == null || files.length == 0) {
			throw new Exception(Messages.ImportUtil_0);
		}
		// Ƿڴ̸Ŀ¼
		if (file.getParentFile() == null) {
			throw new Exception(Messages.ImportUtil_3);
		}

		// ȡĿ
		String projectName = null;
		File projFile = null;
		if (files.length == 1) {
			projFile = files[0];
		} else {
			// more than one .projet file
			throw new Exception(Messages.ImportUtil_4);
		}

		Node node = null;
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document d = builder.parse(new InputSource(new FileInputStream(projFile)));
		NodeList nodeList = d.getElementsByTagName("name");
		if (nodeList.getLength() > 0) {
			node = nodeList.item(0);
		}

		if (files2.length == 0) {
			// no wvproj, parse .project to get name
			projectName = node.getTextContent();
		} else if (files2.length == 1) {
			int index = files2[0].getName().indexOf(".wvproj");
			projectName = files2[0].getName().substring(0, index);
		} else {
			throw new Exception(Messages.ImportUtil_4);
		}

		for (char ch : FORBIDDEN_CHAR) {
			if (projectName.indexOf(ch) != -1) {
				throw new Exception(NLS.bind(Messages.AutoStartLoop_WarningInfo, ch));
			}
		}
		// modify .project
		if (node.getTextContent().equals(projectName)) {
			node.setTextContent(projectName);
		}

		if (files3.length == 1) {
			if (files3[0].getName().endsWith(".template")) {
				BufferedReader br = new BufferedReader(new FileReader(files3[0]));
				String line = null;
				StringBuffer sb = new StringBuffer();
				while ((line = br.readLine()) != null) {
					if (line.startsWith("Target Path=obj\\")) {
						sb.append("Target Path=obj\\" + projectName + ".hex" + NEWLINE);
					} else if (line.startsWith("Target Path=obj/")) {
						sb.append("Target Path=obj/" + projectName + ".hex" + NEWLINE);
					} else {
						sb.append(line + NEWLINE);
					}
				}

				FileWriter fw = new FileWriter(files3[0]);
				fw.write(sb.toString());

				br.close();
				fw.close();

			}
		}

		// Add for MounRiver 1.80
		if (files4.length == 1) {
			// Loaded project has a launch configuration
			if (files4[0].getName().endsWith(".launch")) {
				BufferedReader br = new BufferedReader(new FileReader(files4[0]));
				String line = null;
				StringBuffer sb = new StringBuffer();
				while ((line = br.readLine()) != null) {
					line = line.replace("ilg.gnumcueclipse", "com.mounriver");
					sb.append(line + NEWLINE);
				}
				FileWriter fw = new FileWriter(files4[0]);
				fw.write(sb.toString());
				br.close();
				fw.close();

				// Add for prototype e-mrs
				File newLaunchFile = new File(
						files4[0].getParentFile().getAbsolutePath() + File.separator + projectName + ".launch");
				if (!newLaunchFile.exists()) {
					Files.copy(files4[0].toPath(), newLaunchFile.toPath());
					files4[0].delete();
				}
			}
		}

//		System.out.print("project-->" + projectName + "\r\n"); //$NON-NLS-1$ //$NON-NLS-2$

		// Ŀ¼
		IWorkspace workspace = ResourcesPlugin.getWorkspace();

		IProjectDescription newProjectDescription = workspace.newProjectDescription(projectName);
		IProject newProject = workspace.getRoot().getProject(projectName);

		if (newProject.exists()) {
			if (!(newProject.getLocationURI().getPath().replaceFirst("/", "").replace("/", "\\") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					.equalsIgnoreCase(projLoc))) {

				if (MessageDialog.openQuestion(new Shell(), Messages.ImportUtil_1, Messages.ImportUtil_2)) {
					newProject.delete(IResource.NEVER_DELETE_PROJECT_CONTENT, new NullProgressMonitor());
					importProjectFromFolder(projLoc, monitor, solutionPath);
				} else {

				}
			} else {
				// ͬһĿ¼
				// MessageDialog.openInformation(new Shell(), Messages.ImportUtil_1,
				// Messages.ImportUtil_6);
				if (solutionPath == null) {
					// load proj, not load solution
					setProjActive(newProject);
				}
				newProject.open(null);
				if (files2.length == 0) {
					createWvprojFile(newProject);
				}
				return;
			}
		} else {
//			int index = newProjectLocation.lastIndexOf("\\");
//			String parentPath = newProjectLocation.substring(0, index);

//			if (parentPath.equals(workspacePath)) {
//				newProjectDescription.setLocationURI(null);
//			} else {
//				newProjectDescription.setLocationURI(URIUtil.toURI(newProjectLocation)); // $NON-NLS-1$
//			}
			// Add for MounRiver, shield if virtual mapping
			newProjectDescription.setLocationURI(URIUtil.toURI(projLoc)); // $NON-NLS-1$
			newProject.create(newProjectDescription, null);

			IOverwriteQuery overwriteQuery = new IOverwriteQuery() {
				public String queryOverwrite(String file) {
					return ALL;
				}
			};

			ImportOperation importOperation = new ImportOperation(newProject.getFullPath(), file,
					FileSystemStructureProvider.INSTANCE, overwriteQuery);
			importOperation.setCreateContainerStructure(false);
			importOperation.run(monitor);
			importOperation = null;
			file = null;

			// add for MounRiver 1.5
			if (solutionPath == null) {
				// load proj, not load solution
				setProjActive(newProject);
				newProject.open(null);
			} else {
				// flag project for future purpose
				newProject.open(null);
				newProject.setPersistentProperty(new QualifiedName("projectAttr", "solution"), solutionPath);
			}
			syncLoadProject(newProject, projectName);

			if (files2.length == 0) {
				createWvprojFile(newProject);
			}
		}
	}

	private static void setProjActive(IProject newProject) {
		Display.getDefault().syncExec(new Runnable() {
			@Override
			public void run() {
				IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
				if (windows.length > 0) {
					IWorkbenchPart part = windows[0].getActivePage().findView(ProjectExplorer.VIEW_ID);
					if (part != null) {
						NavigatorContentService contentService = (NavigatorContentService) Adapters.adapt(part,
								INavigatorContentService.class);
						if (contentService != null) {
							CommonViewer viewer = (CommonViewer) contentService.getViewer();
							IProject[] pp = new IProject[] { newProject };
							viewer.setSelection(new TreeSelection(new TreePath(pp)));
							viewer.refresh();
						} else {
							WorkbenchNavigatorPlugin.log(
									"Could not acquire INavigatorContentService from part (\"" + part.getTitle() //$NON-NLS-1$
											+ "\").", //$NON-NLS-1$
									null);
						}
					}
				}
			}
		});
	}

	public static void unZipFile(File zipFile, String unZipPath) {
		System.out.print("unZipFile-->" + zipFile.getAbsolutePath() + " To " + unZipPath + "\r\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		File pathFile = new File(unZipPath);
		if (!pathFile.exists()) {
			pathFile.mkdirs();
		}
		ZipFile zip = null;
		InputStream is = null;
		OutputStream os = null;
		try {
			zip = new ZipFile(zipFile);
			// ѹļ
			for (Enumeration<?> entries = zip.entries(); entries.hasMoreElements();) {
				ZipEntry entry = (ZipEntry) entries.nextElement();
				String zipEntryName = entry.getName();
				is = zip.getInputStream(entry);
				String outPath = (unZipPath + "/" + zipEntryName).replaceAll("\\*", "/"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				// ж·Ƿ,򴴽ļ·
				File file = new File(outPath.substring(0, outPath.lastIndexOf('/')));
				if (!file.exists()) {
					file.mkdirs();
				}
				// жļȫ·ǷΪļ
				if (new File(outPath).isDirectory()) {
					continue;
				}
				// ѹļ
				os = new FileOutputStream(outPath);
				byte[] buf1 = new byte[1024];
				int len;
				while ((len = is.read(buf1)) > 0) {
					os.write(buf1, 0, len);
				}
			}
			zip.close();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
			if (os != null) {
				try {
					os.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	private static final String ALLOWED_CHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_.!~*'()"; //$NON-NLS-1$

	public static String encodeURIComponent(String input) {
		if (input == null || input.equalsIgnoreCase("")) { //$NON-NLS-1$
			return input;
		}

		int l = input.length();
		StringBuilder o = new StringBuilder(l * 3);
		try {
			for (int i = 0; i < l; i++) {
				String e = input.substring(i, i + 1);
				if (ALLOWED_CHARS.indexOf(e) == -1) {
					byte[] b = e.getBytes("utf-8"); //$NON-NLS-1$
					o.append(getHex(b));
					continue;
				}

				o.append(e);
			}
			return o.toString();
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return input;
	}

	private static String getHex(byte buf[]) {
		StringBuilder o = new StringBuilder(buf.length * 3);
		for (int i = 0; i < buf.length; i++) {
			int n = buf[i] & 0xff;
			o.append("%"); //$NON-NLS-1$
			if (n < 0x10) {
				o.append("0"); //$NON-NLS-1$
			}
			o.append(Long.toString(n, 16).toUpperCase());
		}
		return o.toString();
	}

	// synchronize related files in .project-load process
	private static void syncLoadProject(IProject project, String name) throws Exception {
		XmlMofidyUtil.getInstance().onlyModifyProjectName(project, name);

		File file = new File(project.getLocation().toOSString());
		if (file.exists() && file.isDirectory()) {
			File launchFile = null;
			for (File f : file.listFiles()) {
				if (f.getName().endsWith(".launch")) {
					launchFile = f;
					break;
				}
			}
			if (launchFile != null) {
				XmlMofidyUtil.getInstance().modifyLaunchContent(project, launchFile.getAbsolutePath());
			}
		}
	}

	private static boolean modifyTmpInfoFile(IProject project, String vendor, String series, String link,
			String zipTemplatePath) throws Exception {
		String oldLink = "";
		String description = "";
		File zipFile = new File(zipTemplatePath);
		String mcuName = zipFile.getName().substring(0, zipFile.getName().length() - 4);
		String content = XmlMofidyUtil.getInstance().getChipContent(zipFile);
		String[] arr = content.split("Description=");
		if (arr.length == 2) {
			description = arr[1].split(".\n")[0];
		}
		String filename = project.getLocation().toOSString() + File.separator + ".template";
		File file = new File(filename);
		FileWriter fileWriter = null;
		if (file.exists()) {
			// no need to create .template if already existed
			FileReader fileReader = new FileReader(file);
			BufferedReader bufferedReader = new BufferedReader(fileReader);
			StringBuilder sb = new StringBuilder();
			String line = ""; //$NON-NLS-1$
			boolean b0 = false, b1 = false, b2 = false, b3 = false;
			String str0 = "MCU=";
			String str1 = "Target Path=obj\\";
			String str2 = "Target Path=obj/";
			String str3 = "Vendor=";
			String str4 = "Link=";
			while ((line = bufferedReader.readLine()) != null) {
				if (line.startsWith(str0)) {
					line = str0 + mcuName;
					b0 = true;
				}
				if (line.startsWith(str1)) {
					line = str1 + project.getName() + ".hex";
					b1 = true;
				}
				if (line.startsWith(str2)) {
					line = str2 + project.getName() + ".hex";
					b1 = true;
				}

				if (line.startsWith(str3)) {
					line = str3 + vendor;
					b2 = true;
				}

				if (line.startsWith(str4)) {
					oldLink = line.substring(str4.length());
					line = str4 + link;
					b3 = true;
				}
				sb.append(line + NEWLINE);
			}
			if (!b0) {
				sb.append(str0 + mcuName + NEWLINE);
			}

			if (!b1) {
				sb.append(str1 + project.getName() + ".hex" + NEWLINE);
			}
			if (!b2) {
				sb.append(str3 + vendor + NEWLINE);
			}
			if (!b3) {
				sb.append(str4 + link + NEWLINE);
			}

			fileWriter = new FileWriter(file);
			fileWriter.write(sb.toString());

			bufferedReader.close();
			fileReader.close();
		} else {
			file.createNewFile();
			fileWriter = new FileWriter(file);
			fileWriter.write(NEWLINE + "MCU=" + mcuName + NEWLINE + "Mcu Type=" + NEWLINE + "Address=" + NEWLINE
					+ "Target Path=" + NEWLINE + "Erase All=" + NEWLINE + "Program=" + NEWLINE + "Verify=" + NEWLINE
					+ "Reset=" + NEWLINE + "Vendor=" + vendor + NEWLINE + "Link=" + link + NEWLINE + "Toolchain="
					+ NEWLINE + "Series=" + series + NEWLINE + "Description=" + description);
		}
		if (fileWriter != null) {
			fileWriter.flush();
			fileWriter.close();
		}

		return oldLink.equals(link);
	}

	private static void createWvprojFile(IProject project) throws Exception {
		String filename = project.getLocation().toOSString() + File.separator + project.getName() + ".wvproj";
		File file = new File(filename);
		if (file.exists()) {
			file.delete();
		}
		file.createNewFile();
		FileOutputStream outputStream = new FileOutputStream(file);
		Random random = new Random();
		for (int i = 0; i < 202; i++) {
			outputStream.write(new byte[] { (byte) random.nextInt(200) });
		}
		outputStream.close();
	}
}
