/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package mrsdebugger;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PropertyPage;
import org.eclipse.ui.views.properties.PropertySheet;
import org.eclipse.ui.views.properties.PropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;

import autoStart.AutoStartAction;


public class DebuggerPage extends PropertyPage {
	private static final String NEWLINE = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
	private final static String[] OTHER_LINK__ARR = { "Others" };
	private String[] VENDOR_ARR;
	private Combo comboVendor;
	private Combo comboMcu;
	private Combo comboLink;

	private IProject project;
	private String vendor;
//	private String series;
	private String link;
	private String prjMcu;
//	private String errorMsg = "";

	private ArrayList<MRSComponent> list;
	private ArrayList<String> listZipName = new ArrayList<String>();

	@Override
	protected void initializeDialogUnits(Control testControl) {
		list = AutoStartAction.getMrsComponentList();
		super.initializeDialogUnits(testControl);
	}

	@Override
	protected Control createContents(Composite parent) {
		IResource resource = Adapters.adapt(getElement(), IResource.class);
		if (resource instanceof IProject) {
			project = (IProject) resource;
			readDownloadSettings();
			// create a composite with standard margins and spacing
			Composite composite = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout(2, false);
			layout.marginHeight = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
			layout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
			layout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
			layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
			composite.setLayout(layout);
			composite.setLayoutData(new GridData(GridData.FILL_BOTH));
			composite.setFont(parent.getFont());

			Label labelVendor = new Label(composite, SWT.NONE);
			GridData gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			labelVendor.setLayoutData(gd);
			labelVendor.setText(Messages.DebuggerPage_2);

//			Label labelVendorVal = new Label(composite, SWT.NONE);
//			gd = new GridData();
//			labelVendorVal.setLayoutData(gd);
//			labelVendorVal.setText(vendor);

			comboVendor = new Combo(composite, SWT.DROP_DOWN | SWT.BORDER | SWT.READ_ONLY);
			gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			gd.widthHint = 130;
			comboVendor.setLayoutData(gd);
			comboVendor.setEnabled(false);

			Label labelMcu = new Label(composite, SWT.NONE);
			gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			labelMcu.setLayoutData(gd);
			labelMcu.setText(Messages.DebuggerPage_0);

			comboMcu = new Combo(composite, SWT.DROP_DOWN | SWT.BORDER | SWT.READ_ONLY);
			gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			gd.widthHint = 130;
			comboMcu.setLayoutData(gd);

			Label labelLink = new Label(composite, SWT.NONE);
			gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			labelLink.setLayoutData(gd);
			labelLink.setText(Messages.DebuggerPage_1);

			comboLink = new Combo(composite, SWT.DROP_DOWN | SWT.BORDER | SWT.READ_ONLY);
			gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
			gd.widthHint = 130;
			comboLink.setLayoutData(gd);

			initControls();
			addListeners();
		}
		return null;
	}

	private void addListeners() {
		comboVendor.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String newVendor = comboVendor.getText();

				int index = Arrays.asList(VENDOR_ARR).indexOf(newVendor);
				comboVendor.select(index);

				switch (newVendor) {
				case "Others":
					comboMcu.setItems(new String[] {});
//					comboMcu.setEnabled(false);
					comboMcu.select(-1);

					comboLink.setItems(OTHER_LINK__ARR);
					comboLink.select(0);
					break;
				default:
					MRSComponent mrsComponent = null;
					for (MRSComponent ele : list) {
						if (ele.getVendor().equals(newVendor)) {
							mrsComponent = ele;
							break;
						}
					}
					if (mrsComponent != null) {
						ArrayList<MCU> mculist = mrsComponent.getMcuList();
						String[] strMcuArr = new String[mculist.size()];

						String[] strLinkArr = null;
						for (int i = 0; i < mculist.size(); i++) {
							MCU mcu = mculist.get(i);
							strMcuArr[i] = mcu.getName();
						}

//						if (prjMcu != null && !prjMcu.isEmpty() && Arrays.asList(strMcuArr).contains(prjMcu)) {
//							comboMcu.setItems(listZipName.toArray(new String[] {}));
//							comboMcu.select(listZipName.indexOf(prjMcu));
//						} else {
						comboMcu.setItems(strMcuArr);
						comboMcu.select(0);
//						}

						for (int i = 0; i < mculist.size(); i++) {
							MCU mcu = mculist.get(i);
							if (mcu.getName().equals(strMcuArr[0])) {
								ArrayList<Debugger> debuggerList = mcu.getDebuggerList();
								strLinkArr = new String[debuggerList.size()];
								for (int j = 0; j < debuggerList.size(); j++) {
									strLinkArr[j] = debuggerList.get(j).getName();
								}
							}
						}
						comboLink.setItems(strLinkArr);
						comboLink.select(0);
					}
				}

				updateControlStatus();
			}

		});

		comboMcu.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String newVendor = comboVendor.getText();
				String newPrjMcu = comboMcu.getText();
				int index = Arrays.asList(VENDOR_ARR).indexOf(newVendor);
				comboVendor.select(index);

				switch (newVendor) {
				case "Others":
					comboLink.setItems(OTHER_LINK__ARR);
					comboLink.select(0);
					break;
				default:
					// WCH, GD and other vendors
					MRSComponent mrsComponent = null;
					for (MRSComponent ele : list) {
						if (ele.getVendor().equals(newVendor)) {
							mrsComponent = ele;
							break;
						}
					}
					if (mrsComponent != null) {
						ArrayList<MCU> mculist = mrsComponent.getMcuList();

						String[] strLinkArr = null;
						for (int i = 0; i < mculist.size(); i++) {
							MCU mcu = mculist.get(i);
							if (mcu.getName().equals(newPrjMcu)) {
								ArrayList<Debugger> debuggerList = mcu.getDebuggerList();
								strLinkArr = new String[debuggerList.size()];
								for (int j = 0; j < debuggerList.size(); j++) {
									strLinkArr[j] = debuggerList.get(j).getName();
								}
							}
						}
						comboLink.setItems(strLinkArr);
						comboLink.select(0);
					}
				}
				updateControlStatus();
			}
		});

		comboLink.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				updateControlStatus();
			}
		});

	}

	private void initControls() {
		VENDOR_ARR = new String[list.size() + 1];
		for (int i = 0; i < list.size(); i++) {
			MRSComponent mrsComponent = list.get(i);
			VENDOR_ARR[i] = mrsComponent.getVendor();
		}
		VENDOR_ARR[list.size()] = "Others";
		comboVendor.setItems(VENDOR_ARR);

		if (vendor == null || vendor.isEmpty()) {
			comboVendor.select(-1);
			comboVendor.setEnabled(true);

			comboMcu.setItems(new String[] {});
			comboMcu.select(-1);
			comboMcu.setEnabled(true);

			comboLink.setItems(new String[] {});
			comboLink.select(-1);
			comboLink.setEnabled(true);
		} else {
			int index = Arrays.asList(VENDOR_ARR).indexOf(vendor);
			comboVendor.select(index);

			switch (vendor) {
			case "Others":
				comboLink.setItems(OTHER_LINK__ARR);
				comboLink.select(0);
				break;
			default:
				// WCH, GD and other vendors
				MRSComponent mrsComponent = null;
				for (MRSComponent ele : list) {
					if (ele.getVendor().equals(vendor)) {
						mrsComponent = ele;
						break;
					}
				}
				if (mrsComponent != null) {
					ArrayList<MCU> mculist = mrsComponent.getMcuList();
					String[] strMcuArr = new String[mculist.size()];

					String[] strLinkArr = null;
					for (int i = 0; i < mculist.size(); i++) {
						MCU mcu = mculist.get(i);
						strMcuArr[i] = mcu.getName();
						if (mcu.getName().equals(prjMcu)) {
							ArrayList<Debugger> debuggerList = mcu.getDebuggerList();
							strLinkArr = new String[debuggerList.size()];
							for (int j = 0; j < debuggerList.size(); j++) {
								strLinkArr[j] = debuggerList.get(j).getName();
							}
						}
					}

					if (prjMcu != null && Arrays.asList(strMcuArr).contains(prjMcu)) {
						comboMcu.setItems(listZipName.toArray(new String[] {}));
						comboMcu.select(listZipName.indexOf(prjMcu));
					} else {
						comboMcu.setItems(strMcuArr);
						if (prjMcu == null) {
							index = -1;
						} else {
							index = Arrays.asList(strMcuArr).indexOf(prjMcu);
						}
//						if (index == -1) {
//							errorMsg += " " + comboMcu;
//						}
						comboMcu.select(index);
					}

					if (strLinkArr != null) {
						comboLink.setItems(strLinkArr);
						index = Arrays.asList(strLinkArr).indexOf(link);
//						if (index == -1) {
//							errorMsg += " " + link;
//						}
						comboLink.select(index);
					} else {
						comboLink.select(-1);
					}
				}

//				if (!errorMsg.isEmpty()) {
//					errorMsg += " not supported for " + vendor + ". Please check component or reselect debugger.";
//					setMessage(errorMsg, IMessageProvider.WARNING);
//				}
			}
		}
	}

	public void readDownloadSettings() {
//		String path = project.getLocation().toOSString() + File.separator + ".cproject";
//		File cpjFile = new File(path);
//		if (cpjFile.exists()) {
//			try {
//				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
//				DocumentBuilder builder = factory.newDocumentBuilder();
//				Document d = builder.parse(new InputSource(new FileInputStream(cpjFile)));
//				NodeList nodeList = d.getElementsByTagName("configuration");
//				Node node = nodeList.item(0);
//				NamedNodeMap namedNodeMap = node.getAttributes();
//				for (int j = 0; j < namedNodeMap.getLength(); j++) {
//					if (namedNodeMap.item(j).getNodeName().equals("id")) {
//						isARM = namedNodeMap.item(j).getNodeValue().startsWith("ilg.gnuarmeclipse");
//						break;
//					}
//				}
//			} catch (Exception e) {
//				e.printStackTrace();
//			}
//		}

		String templatePath = project.getLocation().toOSString() + File.separator + ".template";
		File templateFile = new File(templatePath);
		if (templateFile.exists()) {
			BufferedReader br = null;
			try {
				br = new BufferedReader(new FileReader(templateFile));
				String line = null;
				while ((line = br.readLine()) != null) {
					if (line.startsWith("Vendor=")) {
						vendor = line.substring("Vendor=".length()).trim();
						continue;
					}
//					else if (line.startsWith("Series=")) {
//						series = line.substring("Series=".length()).trim();
//						continue;
//					}

					else if (line.startsWith("Link=")) {
						link = line.substring("Link=".length()).trim();
						continue;
					}

					else if (line.startsWith("MCU=")) {
						prjMcu = line.substring("MCU=".length()).trim();
						continue;
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
			if (br != null) {
				try {
					br.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}

			if (prjMcu == null || prjMcu.isEmpty()) {
				return;
			}

			boolean isARM = isARMCore(prjMcu);
			File fParentTc = new File(Platform.getInstallLocation().getURL().getFile() + "template" + File.separator
					+ "wizard" + File.separator + vendor + File.separator + (isARM ? "ARM" : "RISC-V"));
			File found = null;
			if (fParentTc.exists() && fParentTc.isDirectory()) {
				for (File fParentSeries : fParentTc.listFiles()) {
					if (fParentSeries.isDirectory()) {
						for (File fParentOSType : fParentSeries.listFiles()) {
							if (fParentOSType.isDirectory() && fParentOSType.getName().equals("NoneOS")) {
								for (File f : fParentOSType.listFiles()) {
									if (f.isFile() && f.getName().equals(prjMcu + ".zip")) {
										found = f;
										break;
									}
								}
							}
						}
					}
				}
			}

			if (found != null) {
				for (File file : found.getParentFile().listFiles()) {
					if (file.getName().endsWith(".zip")) {
						String zipName = file.getName().substring(0, file.getName().length() - 4);
						listZipName.add(zipName);
					}
				}
			}
		}
	}

	private boolean isARMCore(String strPrjMcu) {
		if (strPrjMcu.startsWith("CH32F") || strPrjMcu.equals("CH563") || strPrjMcu.equals("CH579")) {
			return true;
		}
		return false;
	}

	@Override
	public boolean performOk() {
		if (!updateControlStatus()) {
			return false;
		}
		String strVendor = comboVendor.getText();
		String strLink = comboLink.getText();
		String strPrjMcu = comboMcu.getText();

		try {
			writeTemp(project, strVendor, strPrjMcu, strLink);
			AutoStartAction.updateLaunchContent(project, strVendor, strLink);

			updateProperty();

			boolean targetisARM = isARMCore(strPrjMcu);
			File fParentTc = new File(Platform.getInstallLocation().getURL().getFile() + "template" + File.separator
					+ "wizard" + File.separator + strVendor + File.separator + (targetisARM ? "ARM" : "RISC-V"));
			File found = null;
			if (fParentTc.exists() && fParentTc.isDirectory()) {
				for (File fParentSeries : fParentTc.listFiles()) {
					if (fParentSeries.isDirectory()) {
						for (File fParentOSType : fParentSeries.listFiles()) {
							if (fParentOSType.isDirectory() && fParentOSType.getName().equals("NoneOS")) {
								for (File f : fParentOSType.listFiles()) {
									if (f.isFile() && f.getName().equals(strPrjMcu + ".zip")) {
										found = f;
										break;
									}
								}
							}
						}
					}
				}
			}

			if (found != null) {
				for (File file : found.getParentFile().listFiles()) {
					if (file.getName().endsWith(".zip")) {
						String zipName = file.getName().substring(0, file.getName().length() - 4);
						listZipName.add(zipName);
					}
				}

				if (vendor == null && link == null && prjMcu == null) {
					return true;
				}

//				// modify ld
//				String ldPath = LDContentParser.getLDPath(project);
//				File ldFile = new File(ldPath);
//				String ldContent = getLDContent(found, LDContentParser.getLDRelativePath(found));
//				writeLD(ldFile, ldContent);

				vendor = null;
				link = null;
				prjMcu = null;
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	@Override
	public void performApply() {
		if (!updateControlStatus()) {
			return;
		}
		String strVendor = comboVendor.getText();
		String strLink = comboLink.getText();
		String strPrjMcu = comboMcu.getText();

		try {
			writeTemp(project, strVendor, strPrjMcu, strLink);
			AutoStartAction.updateLaunchContent(project, strVendor, strLink);

			updateProperty();

			boolean targetisARM = isARMCore(strPrjMcu);
			File fParentTc = new File(Platform.getInstallLocation().getURL().getFile() + "template" + File.separator
					+ "wizard" + File.separator + strVendor + File.separator + (targetisARM ? "ARM" : "RISC-V"));
			File found = null;
			if (fParentTc.exists() && fParentTc.isDirectory()) {
				for (File fParentSeries : fParentTc.listFiles()) {
					if (fParentSeries.isDirectory()) {
						for (File fParentOSType : fParentSeries.listFiles()) {
							if (fParentOSType.isDirectory() && fParentOSType.getName().equals("NoneOS")) {
								for (File f : fParentOSType.listFiles()) {
									if (f.isFile() && f.getName().equals(strPrjMcu + ".zip")) {
										found = f;
										break;
									}
								}
							}
						}
					}
				}
			}

			if (found != null) {
				for (File file : found.getParentFile().listFiles()) {
					if (file.getName().endsWith(".zip")) {
						String zipName = file.getName().substring(0, file.getName().length() - 4);
						listZipName.add(zipName);
					}
				}

				if (vendor == null && link == null && prjMcu == null) {
					return;
				}

//				// modify ld
//				String ldPath = LDContentParser.getLDPath(project);
//				File ldFile = new File(ldPath);
//				String ldContent = getLDContent(found, LDContentParser.getLDRelativePath(found));
//				writeLD(ldFile, ldContent);

				vendor = null;
				link = null;
				prjMcu = null;
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private boolean updateControlStatus() {
		String strVendor = comboVendor.getText();
		String strLink = comboLink.getText();
		String strPrjMcu = comboMcu.getText();
		if (strVendor == null || strVendor.isEmpty()) {
			setErrorMessage(Messages.DebuggerPage_5);
			setValid(false);
			return false;
		}
		if (!strVendor.equals("Others") && (strPrjMcu == null || strPrjMcu.isEmpty())) {
			setErrorMessage(Messages.DebuggerPage_4);
			setValid(false);
			return false;
		}
		if (strLink == null || strLink.isEmpty()) {
			setErrorMessage(Messages.DebuggerPage_3);
			setValid(false);
			return false;
		}
		setErrorMessage(null);
		setValid(true);
		return true;
	}

	@Override
	protected void performDefaults() {
		vendor = null;
		prjMcu = null;
		link = null;
		setErrorMessage(null);
		setValid(true);
		initControls();
	}

	public void writeTemp(IProject project, String strVendor, String mcu, String link) throws Exception {
		if (project != null) {
			String templatePath = project.getLocation().toOSString() + File.separator + ".template"; //$NON-NLS-1$
			File fileTemp = new File(templatePath);
			boolean bCreate = false;
			if (!fileTemp.exists()) {
				fileTemp.createNewFile();
				bCreate = true;
			}

			BufferedWriter bw = null;
			try {
				StringBuffer stringBuffer = new StringBuffer();
				if (bCreate) {
					bw = new BufferedWriter(new FileWriter(fileTemp));
					bw.write("Toolchain=" + NEWLINE);
					bw.write("Series=" + NEWLINE);
					bw.write("Description=" + NEWLINE);
					bw.write("Address=" + NEWLINE);
					bw.write("Target Path=" + NEWLINE);
					bw.write("Erase All=true" + NEWLINE);
					bw.write("Program=true" + NEWLINE);
					bw.write("Verify=true" + NEWLINE);
					bw.write("Reset=true" + NEWLINE);
					bw.write("Vendor=" + strVendor + NEWLINE);
					bw.write("MCU=" + mcu + NEWLINE);
					bw.write("Mcu Type=" + mcuToType(mcu) + NEWLINE);
					bw.write("Link=" + link);
				} else {
					BufferedReader br = null;
					try {
						br = new BufferedReader(new FileReader(fileTemp));
						String line = null;
						while ((line = br.readLine()) != null) {
							if (!line.startsWith("Vendor=") && !line.startsWith("MCU=") && !line.startsWith("Link=")
									&& !line.startsWith("Mcu Type=")) {
								stringBuffer.append(line + NEWLINE);
							}
						}
					} catch (Exception e) {
						e.printStackTrace();
					} finally {
						try {
							br.close();
							br = null;
						} catch (IOException e) {
							e.printStackTrace();
						}
					}

					bw = new BufferedWriter(new FileWriter(fileTemp));
					if (stringBuffer.length() > 0) {
						bw.write(stringBuffer.toString());
					}
					bw.write(NEWLINE + "Vendor=" + strVendor);
					bw.write(NEWLINE + "MCU=" + mcu);
					bw.write(NEWLINE + "Mcu Type=" + mcuToType(mcu));
					bw.write(NEWLINE + "Link=" + link + NEWLINE);
					bw.flush();
				}

			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				if (bw != null) {
					try {
						bw.close();
						bw = null;
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}

	private static String mcuToType(String mcu) {
		if (mcu != null) {
			if (mcu.startsWith("TWEN-CH32V103") || mcu.startsWith("CH32V10")) {
				return "CH32V10x";
			} else if (mcu.startsWith("CH32V20")) {
				return "CH32V20x";
			} else if (mcu.startsWith("TencentOS-Tiny-CH32V307") || mcu.startsWith("CH32V30")) {
				return "CH32V30x";
			} else if (mcu.startsWith("CH56")) {
				return "CH56x";
			} else if (mcu.startsWith("CH57")) {
				return "CH57x";
			} else if (mcu.startsWith("CH58")) {
				return "CH58x";
			} else if (mcu.startsWith("CH32F10")) {
				return "CH32F10x";
			} else if (mcu.startsWith("CH32F20")) {
				return "CH32F20x";
			}
		}
		return "";
	}

	private void writeLD(File ldFile, String ldContent) throws IOException {
		if (ldFile.exists()) {
			FileWriter fw = new FileWriter(ldFile, false);
			fw.write(ldContent);
			fw.flush();
			fw.close();
		}
	}

	public String getLDContent(File zip, String ldPathRelative) throws Exception {
		StringBuffer stringBuffer = new StringBuffer();
		ZipFile zipFile = new ZipFile(zip.getAbsolutePath()); // 
		Enumeration<?> entries = zipFile.entries();
		while (entries.hasMoreElements()) {
			ZipEntry entry = (ZipEntry) entries.nextElement();
			String filename = entry.getName();
			if (ldPathRelative.replace("/", "\\").contains(filename.replace("/", "\\"))) {
				InputStream is = zipFile.getInputStream(entry);
				byte buf[] = new byte[1024];
				int count = -1;
				while ((count = is.read(buf)) > -1) {
					String str = new String(buf, 0, count);
					stringBuffer.append(str);
				}
			}
		}
		zipFile.close();
		return stringBuffer.toString();
	}
	
	private void updateProperty() throws PartInitException {
		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		String property_viewID = "org.eclipse.ui.views.PropertySheet";
		IViewPart viewPart = page.findView(property_viewID);
		if (viewPart == null) {
			try {
				PropertySheet propertySheet = (PropertySheet) page.showView(property_viewID);
				PropertySheetPage propertySheetPage = (PropertySheetPage) propertySheet.getCurrentPage();
				propertySheetPage.refresh();
			} catch (PartInitException e) {
			}
		} else {
//			page.bringToTop(viewPart);
			PropertySheet propertySheet = (PropertySheet) page.showView(property_viewID);
			TabbedPropertySheetPage propertySheetPage = (TabbedPropertySheetPage) propertySheet.getCurrentPage();
			propertySheetPage.refresh();
		}
	}

}
