/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package systemConfig;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.BaseLabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;

public class MacrosConfigDialog extends Dialog {

	private TableViewer tableViewer;
	private Table table;
	private Text textIntro;
	private ArrayList<Config> list = new ArrayList<Config>();
	private IProject project;

	protected MacrosConfigDialog(Shell parentShell, IProject project) {
		super(parentShell);
		this.project = project;
		setShellStyle(getShellStyle() | SWT.SHEET);
	}

	protected void configureShell(Shell shell) {
		super.configureShell(shell);
		shell.setText("Macros Config");
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		Composite userComp = new Composite(parent, SWT.NONE);
		userComp.setLayout(new GridLayout(1, false));

		Composite topComp = new Composite(userComp, SWT.NONE);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 350;
		topComp.setLayout(new GridLayout());
		topComp.setLayoutData(gd);

		Composite belowComp = new Composite(userComp, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 100;
		belowComp.setLayout(new GridLayout());
		belowComp.setLayoutData(gd);

		tableViewer = new TableViewer(topComp, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.SINGLE);
		table = tableViewer.getTable();

		textIntro = new Text(belowComp, SWT.BORDER | SWT.WRAP | SWT.READ_ONLY | SWT.MULTI | SWT.V_SCROLL);
		gd = new GridData(GridData.FILL_BOTH);
//		gd.heightHint = 100;
		textIntro.setLayoutData(gd);

		tableViewer.getTable().setLinesVisible(true);
		tableViewer.getTable().setHeaderVisible(true);
		addColumns();
		

		tableViewer.getTable().addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDown(MouseEvent e) {
				Point point = new Point(e.x, e.y);
				TableItem item = table.getItem(point);
				if (item != null) {
					Object obj = item.getData();
					if (obj instanceof Config) {
						Config config = (Config) obj;
						textIntro.setText("Macro:[" + config.getName() + "]");
					}
				}
			}
		});
		tableViewer.setContentProvider(new IStructuredContentProvider() {
			@Override
			public Object[] getElements(Object inputElement) {
				if (inputElement instanceof ArrayList) {
					ArrayList arrayList = (ArrayList) inputElement;
					return arrayList.toArray();
				}
				return new Object[0];
			}
		});
		tableViewer.setLabelProvider(new TableLabelProvider() {
			@Override
			public String getColumnText(Object element, int columnIndex) {
				if (element instanceof Config) {
					Config config = (Config) element;
					if (columnIndex == 0) {
						String str = Config.getConfigNameDisplayMap().get(config.getName());
						if (str != null) {
							return str;
						}
						return config.getName();
					} else if (columnIndex == 1) {
						String val = config.getValue();
						if (config.isOnlyDefine()) {
							TableEditor editor = new TableEditor(table);
							editor.horizontalAlignment = SWT.LEFT;
							editor.grabHorizontal = true;
							editor.minimumWidth = 50;

							TableItem currentItem = null;
							for (int i = 0; i < table.getItemCount(); i++) {
								TableItem tableItem = table.getItem(i);
								if (tableItem.getData().equals(element)) {
									currentItem = tableItem;
									break;
								}
							}

							if (val == null) {
								Button btn = new Button(table, SWT.CHECK);
								btn.setFocus();
								btn.setSelection(true);
								editor.setEditor(btn, currentItem, 1);
							} else {
//								Button btn = new Button(table, SWT.CHECK);
//								btn.setFocus();
//								btn.setSelection(true);
//								editor.setEditor(btn, currentItem, 1);
							}
						} else {
							return val;
						}
					}
				}
				return "";
			}
		});

		BufferedReader br = null;

		try {
			File file = new File(project.getLocation().toOSString() + File.separator + "config.h");
			if (file.exists()) {
				br = new BufferedReader(new FileReader(file));
				String line = null;
//			boolean bStartRecord = false;
				boolean bStartRecord = true;
				while ((line = br.readLine()) != null) {
//				if (line.contains(START_LINE)) {
//					bStartRecord = true;
//				}

					if (bStartRecord && line.startsWith("#define ")) {
						Config config = null;
						String arr[] = line.split(" ");
						if (arr.length == 2) {
							config = new Config(arr[1], null, null, true);
						} else {
							config = new Config(arr[1], arr[2], null, false);
						}
						list.add(config);
					}
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (br != null) {
				try {
					br.close();
				} catch (IOException e1) {
					e1.printStackTrace();
				}
			}
		}

		tableViewer.setInput(list);
		table.setLayoutData(new GridData(GridData.FILL_BOTH));

		return userComp;
	}

	private void addColumns() {
		Table table = tableViewer.getTable();
		TableColumn column0 = new TableColumn(table, SWT.NONE);
		column0.setText("Property");
		column0.setResizable(false);
		column0.setWidth(250);

		TableColumn column1 = new TableColumn(table, SWT.NONE);
		column1.setText("Value");
		column1.setResizable(false);
		column1.setWidth(310);
	}

	private static abstract class TableLabelProvider extends BaseLabelProvider implements ITableLabelProvider {
		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
	}

	public void refreshText(String editorText) {
		if (tableViewer != null) {
			list.clear();
//			boolean bStartRecord = false;
			boolean bStartRecord = true;
			String[] lines = editorText.split("\r\n");
			for (int i = 0; i < lines.length; i++) {
				String line = lines[i];
//				if (line.contains(START_LINE)) {
//					bStartRecord = true;
//				}

				if (bStartRecord && line.startsWith("#define ")) {
					Config config = null;
					String arr[] = line.split(" ");
					if (arr.length == 2) {
						config = new Config(arr[1], null, null, true);
					} else {
						config = new Config(arr[1], arr[2], null, false);
					}
					list.add(config);
				}
			}
			tableViewer.refresh();
		}
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, "OK", true);
		createButton(parent, IDialogConstants.CANCEL_ID, "Cancel", false);
//		super.createButtonsForButtonBar(parent);
	}
}
