/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package template;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;

import convertProjectUtil.XmlMofidyUtil;
import handlers.Messages;
import wizard.FileTool;

@SuppressWarnings("restriction")
public class TemplateExportHandler extends AbstractHandler {
	private static final String NEWLINE = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
	private static final String EMPTY_STR = "";
	private String vendor = EMPTY_STR;
	private String toolchain = EMPTY_STR;
	private String series = EMPTY_STR;
	private String rtos = EMPTY_STR;
	private String description = EMPTY_STR;
	private String templateName = EMPTY_STR;
	private String targetPath = EMPTY_STR;
	private Shell shell;

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		IProject project = getFirstProject(event);
		if (project != null) {
			cleanCache();
			parseTemplateInfo(project);
			showDialog(project);
		} else {
			MessageDialog.openInformation(shell, Messages.ExportTemplateHandler_0, Messages.ExportTemplateHandler_2);
		}
		return null;
	}

	private void cleanCache() {
		vendor = EMPTY_STR;
		toolchain = EMPTY_STR;
		series = EMPTY_STR;
		rtos = EMPTY_STR;
		description = EMPTY_STR;
	}

	private void parseTemplateInfo(IProject project) {
		String tmpInfoPath = project.getLocation().toOSString() + File.separator + ".template"; //$NON-NLS-1$
		File tmpFile = new File(tmpInfoPath);
		if (!tmpFile.exists()) {
			return;
		}

		BufferedReader br = null;
		try {
			br = new BufferedReader(new FileReader(tmpFile));
			String line = null;
			while ((line = br.readLine()) != null) {
				int index = line.indexOf("Vendor"); //$NON-NLS-1$
				if (index != -1) {
					vendor = line.substring("Vendor".length() + 1); //$NON-NLS-1$
					continue;
				}

				index = line.indexOf("Toolchain"); //$NON-NLS-1$
				if (index != -1) {
					toolchain = line.substring("Toolchain".length() + 1); //$NON-NLS-1$
					continue;
				}

				index = line.indexOf("Series"); //$NON-NLS-1$
				if (index != -1) {
					series = line.substring("Series".length() + 1); //$NON-NLS-1$
					continue;
				}

				index = line.indexOf("RTOS"); //$NON-NLS-1$
				if (index != -1) {
					rtos = line.substring("RTOS".length() + 1); //$NON-NLS-1$
					continue;
				}

				index = line.indexOf("Description"); //$NON-NLS-1$
				if (index != -1) {
					description = line.substring("Description".length() + 1); //$NON-NLS-1$
					description = description.replaceAll("\\n", "\n");
					continue;
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				br.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	private IProject getFirstProject(ExecutionEvent event) {
		IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindow(event);
		IWorkbenchPage activePage = window.getActivePage();
		shell = activePage.getActivePart().getSite().getShell();
		ISelection selection = activePage.getSelection();
		if (selection instanceof TreeSelection) {
			TreeSelection treeSelection = (TreeSelection) selection;
			Object element = treeSelection.getFirstElement();
			if (element instanceof IResource) {
				return ((IResource) element).getProject();
			} else if (element instanceof ICElement) {
				ICProject cProject = ((ICElement) element).getCProject();
				return cProject.getProject();
			} else if (element instanceof IEditorPart) {
				IFile file = (IFile) (((IEditorPart) element).getEditorInput().getAdapter(IFile.class));
				return file.getProject();
			}
		}
		return null;
	}

	private void showDialog(IProject project) {
		ProjectExportAsTemplateDialog dialog = new ProjectExportAsTemplateDialog(shell, vendor, toolchain, series, rtos,
				description, project.getName());
		if (dialog.open() == Window.OK) {
			templateName = project.getName();
			targetPath = dialog.getPath();
			description = dialog.getDescription();

			File source = new File(project.getLocation().toOSString());
			if (source.exists() && source.isDirectory()) {
				File targetZip = new File(targetPath + File.separator + templateName + ".zip");
				if (targetZip.exists()) {
					boolean result = MessageDialog.openQuestion(shell, Messages.TemplateExportHandler_1,
							Messages.TemplateExportHandler_0);
					if (!result) {
						return;
					}
				}
				// create target vendor directory
				File target = new File(targetPath + File.separator + templateName);
				if (!target.exists()) {
					target.mkdirs();
				}

				addFileRecursive(source.getAbsolutePath(), target.getAbsolutePath(), source, target);

				// modify chip information file
				File fileTemp = new File(targetPath + File.separator + templateName + File.separator + ".template");
				if (!fileTemp.exists()) {
					try {
						fileTemp.createNewFile();
					} catch (IOException e1) {
						e1.printStackTrace();
					}
				}
				modifyTempInfo(fileTemp, dialog.getVendor(), dialog.getToolchain(), dialog.getSeries(),
						dialog.getRTOS(), dialog.getDescription());

				// compress target zip. Deprecated: export vendor dir.
				try {
					FileTool.compress(target.getAbsolutePath(), "zip"); //$NON-NLS-1$
					FileTool.deleteFolder(target.getAbsolutePath());
				} catch (Exception e) {
					e.printStackTrace();
					MessageDialog.openInformation(shell, Messages.ExportTemplateHandler_10,
							Messages.ExportTemplateHandler_11 + e.getMessage());
					return;
				}
				MessageDialog.openInformation(shell, Messages.ExportTemplateHandler_12,
						Messages.ExportTemplateHandler_13);
			}
		}
	}

	/**
	 * Add chip information
	 * 
	 * @param fileTemp
	 * @param vendor
	 * @param series
	 * @param description
	 */

	private void modifyTempInfo(File fileTemp, String vendor, String toolchain, String series, String rtos,
			String description) {
		try {
			FileReader fr = new FileReader(fileTemp);
			int ch;
			StringBuffer stringBuffer = new StringBuffer();
			while ((ch = fr.read()) != -1) {
				stringBuffer.append((char) ch);
			}
			fr.close();
			String strContent = stringBuffer.toString();
			Pattern pattern0 = Pattern
					.compile(Platform.getOS().equals(Platform.OS_WIN32) ? ".*Vendor=(.*)\r\n.*" : ".*Vendor=(.*)\n.*");
			Matcher matcher0 = pattern0.matcher(strContent);
			if (matcher0.find()) {
				strContent = strContent.replace("Vendor=" + matcher0.group(1), "Vendor=" + vendor);
			} else {
				String str = "Vendor=" + vendor + NEWLINE;
				if (strContent.endsWith("\r\n")) {
					strContent += str;
				} else {
					strContent += "\r\n";
					strContent += str;
				}
			}

			Pattern pattern1 = Pattern.compile(
					Platform.getOS().equals(Platform.OS_WIN32) ? ".*Toolchain=(.*)\r\n.*" : ".*Toolchain=(.*)\n.*");
			Matcher matcher1 = pattern1.matcher(strContent);
			if (matcher1.find()) {
				strContent = strContent.replace("Toolchain=" + matcher1.group(1), "Toolchain=" + toolchain);
			} else {
				String str = "Toolchain=" + toolchain + NEWLINE;
				if (strContent.endsWith("\r\n")) {
					strContent += str;
				} else {
					strContent += "\r\n";
					strContent += str;
				}
			}

			Pattern pattern2 = Pattern
					.compile(Platform.getOS().equals(Platform.OS_WIN32) ? ".*Series=(.*)\r\n.*" : ".*Series=(.*)\n.*");
			Matcher matcher2 = pattern2.matcher(strContent);
			if (matcher2.find()) {
				strContent = strContent.replace("Series=" + matcher2.group(1), "Series=" + series);
			} else {
				String str = "Series=" + series + NEWLINE;
				if (strContent.endsWith("\r\n")) {
					strContent += str;
				} else {
					strContent += "\r\n";
					strContent += str;
				}
			}

			Pattern pattern4 = Pattern
					.compile(Platform.getOS().equals(Platform.OS_WIN32) ? ".*RTOS=(.*)\r\n.*" : ".*RTOS=(.*)\n.*");
			Matcher matcher4 = pattern4.matcher(strContent);
			if (matcher4.find()) {
				strContent = strContent.replace("RTOS=" + matcher4.group(1), "RTOS=" + rtos);
			} else {
				String str = "RTOS=" + rtos + NEWLINE;
				if (strContent.endsWith("\r\n")) {
					strContent += str;
				} else {
					strContent += "\r\n";
					strContent += str;
				}
			}

			Pattern pattern3 = Pattern.compile(
					Platform.getOS().equals(Platform.OS_WIN32) ? ".*Description=(.*)\r\n.*" : ".*Description=(.*)\n.*");
			Matcher matcher3 = pattern3.matcher(strContent);
			if (matcher3.find()) {
				description = description.replaceAll("\r\n","\n");
				strContent = strContent.replace("Description=" + matcher3.group(1), "Description=" + description);
			} else {
				String str = "Description=" + description + NEWLINE;
				if (strContent.endsWith("\r\n")) {
					strContent += str;
				} else {
					strContent += "\r\n";
					strContent += str;
				}
			}

			FileWriter fileWriter = new FileWriter(fileTemp);
			if (strContent.isEmpty()) {
				fileWriter.write(NEWLINE + "Vendor=" + vendor + NEWLINE + "Series=" + series + NEWLINE + "RTOS=" + rtos
						+ NEWLINE + "Description=" + description + NEWLINE);
			} else {
				fileWriter.write(strContent);
			}
			fileWriter.flush();
			fileWriter.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private void addFileRecursive(String sourceRoot, String targetRoot, File source, File target) {
		for (File file : source.listFiles()) {
			if (file.isDirectory()) {
				if (file.getName().equals(".settings") || file.getName().equals("obj")) { //$NON-NLS-1$
					// ignore setting and output directory
					continue;
				}
				String relativePath = file.getAbsolutePath().replace(sourceRoot, ""); //$NON-NLS-1$
				File newTarget = new File(targetRoot + relativePath);
				newTarget.mkdirs();
				addFileRecursive(sourceRoot, targetRoot, file, newTarget);
			} else if (file.isFile()) {
				if (file.getName().endsWith("wvproj")) { //$NON-NLS-1$ //$NON-NLS-2$
					// ignore .wvproj, it will be automatically created when import
					continue;
				}
				String relativePath = file.getAbsolutePath().replace(sourceRoot, ""); //$NON-NLS-1$
				File targetFile = new File(targetRoot + relativePath);
				try {
					Files.copy(file.toPath(), targetFile.toPath());
					if (file.getName().equals(".project")) { //$NON-NLS-1$
						XmlMofidyUtil.getInstance().onlyModifyProjectName(targetFile, templateName);
					} else if (file.getName().equals(".cproject")) { //$NON-NLS-1$
						XmlMofidyUtil.getInstance().modifyNewCProjectWithProject(targetFile, templateName);
					} else if (file.getName().endsWith(" obj.launch")) { //$NON-NLS-1$
						XmlMofidyUtil.getInstance().modifyLaunchContent(targetFile, templateName);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
}
