/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package template;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeCursor;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.wb.swt.ResourceManager;

import convertProjectUtil.XmlMofidyUtil;

public class TemplateManageDialog extends TitleAreaDialog {
	private static final String EMPTY_STR = "";
	private static Image imgExpandAll = ResourceManager.getPluginImage("com.mounriver.project", "icon/expandall.gif"); //$NON-NLS-1$ //$NON-NLS-2$
	private static Image imgCollapseAll = ResourceManager.getPluginImage("com.mounriver.project", //$NON-NLS-1$
			"icon/collapseall.gif"); //$NON-NLS-1$
	private static Pattern pattern = Pattern.compile(".*_([0-9])+.*"); //$NON-NLS-1$

	private String title;
	private final String headerTitle;
	private final String message;
	private TreeViewer treeViewer;
	private Tree tree;
	private ArrayList<File> inputList = new ArrayList<File>();
	private Button btnImport;
	private Button btnExport;
	private Button btnCopy;
	private Button btnDelete;
	private ToolItem itemCollapseAll;
	private ToolItem itemExpandAll;
//	private Text textFilter;
	private TemplateContentProvider templateContentProvider;
	private TemplateLabelProvider templateLabelProvider;
	private Text textVendor;
	private Text textSeries;
	private Text textRTType;
	private Text textType;
	private Text textDetail;

	public TemplateManageDialog(Shell parentShell) {
		this(parentShell, Messages.TemplateManageDialog_5, Messages.TemplateManageDialog_5,
				Messages.TemplateManageDialog_7); // $NON-NLS-1$
		setShellStyle(getShellStyle() | SWT.SHEET);
	}

	public TemplateManageDialog(Shell parentShell, String title, String headerTitle, String message) {
		super(parentShell);
		this.title = title;
		this.headerTitle = headerTitle;
		this.message = message;

		templateContentProvider = new TemplateContentProvider();
		templateLabelProvider = new TemplateLabelProvider();
		setShellStyle(getShellStyle());
	}

	/**
	 * Creates the OK and Cancel buttons.
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.CANCEL_ID, Messages.TemplateManageDialog_8, false);
	}

	protected void configureShell(Shell shell) {
		super.configureShell(shell);
		shell.setText(this.title);
//		shell.setSize(750, 730);
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		setTitle(this.headerTitle);
		setMessage(this.message);

		// top level composite
		Composite parentComposite = (Composite) super.createDialogArea(parent);

		Composite userComp = new Composite(parentComposite, SWT.FILL);
		GridLayout layout = new GridLayout(2, false);
		layout.marginHeight = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
		layout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
		layout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
		layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
		userComp.setLayout(layout);

		GridData gd = new GridData(GridData.FILL_BOTH);
		userComp.setLayoutData(gd);
		userComp.setFont(parentComposite.getFont());

		// left
		Group groupLeft = new Group(userComp, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		groupLeft.setLayoutData(gd);
		groupLeft.setLayout(new GridLayout(5, false));

		btnImport = new Button(groupLeft, SWT.NONE);
		btnImport.setText(Messages.TemplateManageDialog_9);
		btnImport.setEnabled(true);

		btnExport = new Button(groupLeft, SWT.NONE);
		btnExport.setText(Messages.TemplateManageDialog_15);
		btnExport.setEnabled(false);

		btnCopy = new Button(groupLeft, SWT.NONE);
		btnCopy.setText(Messages.TemplateManageDialog_10);
		btnCopy.setEnabled(false);

		btnDelete = new Button(groupLeft, SWT.NONE);
		btnDelete.setText(Messages.TemplateManageDialog_11);
		btnDelete.setEnabled(false);

		ToolBar toolbar = new ToolBar(groupLeft, SWT.FLAT);
		gd = new GridData(SWT.END, SWT.END, false, false);
		toolbar.setLayoutData(gd);

		itemExpandAll = new ToolItem(toolbar, SWT.NONE);
		itemExpandAll.setImage(imgExpandAll);
		itemExpandAll.setHotImage(imgExpandAll);
		itemExpandAll.setDisabledImage(imgExpandAll);
		itemExpandAll.setToolTipText(Messages.TemplateManageDialog_12);

		itemCollapseAll = new ToolItem(toolbar, SWT.NONE);
		itemCollapseAll.setImage(imgCollapseAll);
		itemCollapseAll.setHotImage(imgCollapseAll);
		itemCollapseAll.setDisabledImage(imgCollapseAll);
		itemCollapseAll.setToolTipText(Messages.TemplateManageDialog_13);

		Composite compFilter = new Composite(groupLeft, SWT.NONE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 4;
		compFilter.setLayoutData(gd);
		compFilter.setLayout(new GridLayout(2, false));

//		Label labelFilter = new Label(compFilter, SWT.BOLD);
//		gd = new GridData(GridData.BEGINNING);
//		labelFilter.setLayoutData(gd);
//		labelFilter.setText(Messages.TemplateManageDialog_1);
//
//		textFilter = new Text(compFilter, SWT.BORDER);
//		gd = new GridData();
//		gd.widthHint = 190;
//		textFilter.setLayoutData(gd);
//		textFilter.setFocus();

		treeViewer = new TreeViewer(groupLeft, 68352);
		tree = treeViewer.getTree();
		gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 5;
		//avoid cover whole screen
		gd.heightHint = 400;
		tree.setLayoutData(gd);
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		addColumns();
		treeViewer.setContentProvider(templateContentProvider);
		treeViewer.setLabelProvider(templateLabelProvider);
		treeViewer.setInput(inputList);
		
		TreeCursor cursor = new TreeCursor(tree, SWT.NONE);
		cursor.setBackground(tree.getDisplay().getSystemColor(SWT.COLOR_LIST_SELECTION));
		cursor.setForeground(tree.getDisplay().getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT));
		cursor.setLayout(new FillLayout());
		Menu menu = new Menu(tree);
		MenuItem menuItem = new MenuItem(menu, SWT.PUSH);
		menuItem.setText("Hello World");
		menuItem.setEnabled(true);
		cursor.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				int column = cursor.getColumn();
				if(column == 0) {
					tree.setMenu(menu);
				} else {
					tree.setMenu(null);
				}
			}
		});
		

		// right
		Group groupRight = new Group(userComp, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		groupRight.setLayoutData(gd);
		groupRight.setLayout(new GridLayout(2, false));

		Label labelVendor = new Label(groupRight, SWT.NONE);
//		labelVendor.setForeground(tagColor);
		gd = new GridData();
		labelVendor.setLayoutData(gd);
		labelVendor.setText(Messages.TemplateManageDialog_2);

		textVendor = new Text(groupRight, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData();
		gd.widthHint = 160;
		textVendor.setLayoutData(gd);
		textVendor.setText(EMPTY_STR);

		Label labelSeries = new Label(groupRight, SWT.NONE);
		gd = new GridData();
		labelSeries.setLayoutData(gd);
		labelSeries.setText(Messages.TemplateManageDialog_3);

		textSeries = new Text(groupRight, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData();
		gd.widthHint = 160;
		textSeries.setLayoutData(gd);
		textSeries.setText(EMPTY_STR);

		Label labelRTType = new Label(groupRight, SWT.NONE);
		gd = new GridData();
		labelRTType.setLayoutData(gd);
		labelRTType.setText(Messages.TemplateManageDialog_31);

		textRTType = new Text(groupRight, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData();
		gd.widthHint = 160;
		textRTType.setLayoutData(gd);
		textRTType.setText(EMPTY_STR);

		Label labelType = new Label(groupRight, SWT.NONE);
		gd = new GridData();
		labelType.setLayoutData(gd);
		labelType.setText(Messages.TemplateManageDialog_4);

		textType = new Text(groupRight, SWT.BORDER | SWT.READ_ONLY);
		gd = new GridData();
		gd.widthHint = 160;
		textType.setLayoutData(gd);
		textType.setText(EMPTY_STR);

		Label labelDesc = new Label(groupRight, SWT.NONE);
		gd = new GridData();
		gd.horizontalSpan = 2;
		labelDesc.setLayoutData(gd);
		labelDesc.setText(Messages.TemplateManageDialog_14);

		textDetail = new Text(groupRight, SWT.BORDER | SWT.MULTI | SWT.WRAP | SWT.V_SCROLL | SWT.READ_ONLY);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		textDetail.setLayoutData(gd);
		textDetail.setText(EMPTY_STR);

		addActions();
		refreshTree(false);

		setHelpAvailable(false);
		Dialog.applyDialogFont(parentComposite);
		
		return parentComposite;
	}

	private void refreshTree(boolean expand) {
		File dirWizard = new File(
				Platform.getInstallLocation().getURL().getFile() + "template" + File.separator + "wizard"); //$NON-NLS-1$ //$NON-NLS-2$
		if (dirWizard.exists() && dirWizard.isDirectory()) {
			inputList.clear();
			File[] fileArray = dirWizard.listFiles();
			for (File file : fileArray) {
				inputList.add(file);
			}
			treeViewer.refresh();
			if (expand)
				treeViewer.expandAll();
			else
				treeViewer.collapseAll();
		}
	}

	private void addColumns() {
		TreeColumn treeColumn = new TreeColumn(tree, SWT.NONE);
		treeColumn.setText(Messages.TemplateManageDialog_0);
		treeColumn.setResizable(false);
		treeColumn.setWidth(270);
	}

	private void addActions() {
		tree.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				updateControls();
			}
		});

//		textFilter.addModifyListener(new ModifyListener() {
//			@Override
//			public void modifyText(ModifyEvent e) {
//				String filter = textFilter.getText();
//				templateContentProvider.setFilter(filter);
//				refreshTree(true);
//
//				// select first node
//				TreeItem item = null;
//				if (tree.getItemCount() > 0) {
//					item = tree.getItems()[0];
//					while (item.getItemCount() > 0) {
//						item = item.getItems()[0];
//						tree.setSelection(item);
//					}
//				}
//				if (item != null) {
//					tree.setSelection(item);
//					updateControls();
//				}
//			}
//		});

		itemCollapseAll.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				treeViewer.collapseAll();
			}
		});
		itemExpandAll.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				treeViewer.expandAll();
			}
		});

		tree.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeItem item = (TreeItem) e.item;
				Object data = item.getData();
				if (data instanceof File) {
					File file = (File) data;
					if (file.getName().endsWith(".zip")) { //$NON-NLS-1$
						btnCopy.setEnabled(true);
						btnDelete.setEnabled(true);
						btnImport.setEnabled(true);
						btnExport.setEnabled(true);
					} else {
						btnCopy.setEnabled(false);
						btnDelete.setEnabled(false);
						btnImport.setEnabled(true);
						btnExport.setEnabled(false);
					}
				} else {
					btnCopy.setEnabled(false);
					btnDelete.setEnabled(false);
					btnImport.setEnabled(false);
					btnExport.setEnabled(false);
				}
			}
		});

		btnImport.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TemplateImportDialog dialog = new TemplateImportDialog(getShell());
				if (dialog.open() == Window.OK) {
					String zipPath = dialog.getZipPath();
					File zipFile = new File(zipPath);
					if (zipFile.exists()) {
						String content = null;
						try {
							content = XmlMofidyUtil.getInstance().getChipContent(zipFile);
						} catch (Exception e2) {
							e2.printStackTrace();
						}
						if (content == null)
							return;
						String tempVendor = content.substring(content.indexOf("Vendor=") + "Vendor=".length(),
								content.indexOf("\r\n", content.indexOf("Vendor=") + "Vendor=".length()));
						String tempToolchain = content.substring(content.indexOf("Toolchain=") + "Toolchain=".length(),
								content.indexOf("\r\n", content.indexOf("Toolchain=") + "Toolchain=".length()));
						String tempSeries = content.substring(content.indexOf("Series=") + "Series=".length(),
								content.indexOf("\r\n",content.indexOf("Series=") + "Series=".length()));
						String tempRTOS = content.substring(content.indexOf("RTOS=") + "RTOS=".length(),
								content.indexOf("\r\n",content.indexOf("RTOS=") + "RTOS=".length()));

						String templateFolder = Platform.getInstallLocation().getURL().getFile() + "template" //$NON-NLS-1$
								+ File.separator + "wizard"; //$NON-NLS-1$

						File targetZip = new File(templateFolder + File.separator + tempVendor + File.separator
								+ tempToolchain + File.separator + tempSeries + File.separator + tempRTOS + File.separator + zipFile.getName());
						if (targetZip.exists()) {
							boolean result = MessageDialog.openQuestion(getShell(), Messages.TemplateManageDialog_20,
									targetZip.getAbsolutePath() + " " + Messages.TemplateManageDialog_21);
							if (result) {
								targetZip.delete();
							} else {
								return;
							}
						}
						try {
							File newSeriesFile = new File(templateFolder + File.separator + tempVendor + File.separator
									+ tempToolchain + File.separator + tempSeries + File.separator + tempRTOS);
							newSeriesFile.mkdirs();
							Files.copy(zipFile.toPath(), targetZip.toPath());
							MessageDialog.openInformation(getShell(), Messages.TemplateManageDialog_22,
									Messages.TemplateManageDialog_23);
							refreshTree(true);
							setSelection(targetZip);
							updateControls();

						} catch (IOException e1) {
							e1.printStackTrace();
						}
					}
				}
			}
		});

		btnExport.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TemplateExportDialog templateExportDialog = new TemplateExportDialog(getShell());
				if (Window.OK == templateExportDialog.open()) {
					String exportPath = templateExportDialog.getExportPath();
					if (exportPath != null) {
						ISelection selection = treeViewer.getSelection();
						TreeSelection treeSelection = (TreeSelection) selection;
						Object obj = treeSelection.getFirstElement();
						if (obj instanceof File) {
							File zipFile = (File) obj;
							File desFile = new File(exportPath + File.separator + zipFile.getName());
							if (desFile.exists()) {
								boolean result = MessageDialog.openQuestion(getShell(),
										Messages.TemplateManageDialog_20,
										desFile.getAbsolutePath() + " " + Messages.TemplateManageDialog_21);
								if (result) {
									desFile.delete();
								} else {
									return;
								}
							}
							try {
								Files.copy(zipFile.toPath(), desFile.toPath());
								MessageDialog.openInformation(getShell(), Messages.TemplateManageDialog_22,
										Messages.TemplateManageDialog_16);
							} catch (IOException e1) {
								e1.printStackTrace();
							}
						}
					}
				}
			}
		});

		btnCopy.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ISelection selection = treeViewer.getSelection();
				TreeSelection treeSelection = (TreeSelection) selection;
				Object obj = treeSelection.getFirstElement();
				if (obj == null) {
					btnCopy.setEnabled(false);
					btnDelete.setEnabled(false);
					return;
				}
				if (obj instanceof File) {
					File file = (File) obj;
					File fRT = file.getParentFile();
					File fSeries = fRT.getParentFile();
					File fTc = fSeries.getParentFile();
					File fVendor = fTc.getParentFile();

					String originVendor = fVendor.getName();
					String originTc = fTc.getName();
					String originSeries = fSeries.getName();
					String originRTtype = fRT.getName();
					String origintype = file.getName().replace(".zip", ""); //$NON-NLS-1$ //$NON-NLS-2$
					String type = origintype;
					int start = 1;
					Matcher matcher = pattern.matcher(type);
					if (matcher.find()) {
						int tag = Integer.parseInt(matcher.group(1));
						start = tag + 1;
						type = type.substring(0, type.lastIndexOf("_" + tag)); //$NON-NLS-1$
					}

					String str = file.getAbsolutePath();
					while (true) {
						str = file.getParent() + File.separator + type + "_" + start + ".zip"; //$NON-NLS-1$ //$NON-NLS-2$
						if (!new File(str).exists())
							break;
						start++;
					}
					type = type + "_" + start; //$NON-NLS-1$

					TemplateCopyDialog copyTemplateDialog = new TemplateCopyDialog(getShell(),
							Messages.TemplateManageDialog_35, Messages.TemplateManageDialog_36, originVendor, originTc,
							originSeries, originRTtype, type, origintype);
					if (copyTemplateDialog.open() == Window.OK) {
						String newVendor = copyTemplateDialog.getVendor();
						String newTc = copyTemplateDialog.getToolchain();
						String newSeries = copyTemplateDialog.getSeries();
						String newRTType = copyTemplateDialog.getRttype();
						type = copyTemplateDialog.getType();
						String rtTypePath = Platform.getInstallLocation().getURL().getFile() + "template" //$NON-NLS-1$
								+ File.separator + "wizard" + File.separator + newVendor + File.separator + newTc //$NON-NLS-1$
								+ File.separator + newSeries + File.separator + newRTType;
						File fileRtType = new File(rtTypePath);
						if (!fileRtType.exists()) {
							fileRtType.mkdirs();
						}
						String typePath = fileRtType + File.separator + type + ".zip"; //$NON-NLS-1$
						File fileType = new File(typePath);
						if (fileType.exists()) {
							fileType.delete();
						}
						if (file.exists()) {
							try {
								XmlMofidyUtil.getInstance().modifyChipContent(file, fileType, originVendor, originTc,
										originSeries, newVendor, newTc, newSeries);
							} catch (Exception e1) {
								e1.printStackTrace();
							}
							refreshTree(true);
							setSelection(fileType);
							updateControls();
						}
					}
				}
			}
		});

		btnDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ISelection selection = treeViewer.getSelection();
				TreeSelection treeSelection = (TreeSelection) selection;
				Object obj = treeSelection.getFirstElement();
				if (obj == null) {
					btnCopy.setEnabled(false);
					btnDelete.setEnabled(false);
					return;
				}
				if (obj instanceof File) {
					File fileType = (File) obj;
					String templateName = fileType.getName().replace(".zip", ""); //$NON-NLS-1$ //$NON-NLS-2$
					boolean result = MessageDialog.openQuestion(getShell(), Messages.TemplateManageDialog_11,
							Messages.TemplateManageDialog_27 + templateName + " " + Messages.TemplateManageDialog_28
									+ fileType.getAbsolutePath());
					if (result) {
						File fileRTType = fileType.getParentFile();
						File fileSeries = fileRTType.getParentFile();
						File fileToolChain = fileSeries.getParentFile();
						File fileVendor = fileToolChain.getParentFile();
						fileType.delete();
						if (fileRTType.listFiles().length == 0) {
							fileRTType.delete();
						}
						if (fileSeries.listFiles().length == 0) {
							fileSeries.delete();
						}
						if (fileToolChain.listFiles().length == 0) {
							fileToolChain.delete();
						}
						if (fileVendor.listFiles().length == 0) {
							fileVendor.delete();
						}
						refreshTree(true);
						TreeItem item = null;
						item = tree.getItem(0);
						while(item.getItemCount() > 0) {
							item = item.getItem(0);
						}
						tree.setSelection(item);
						updateControls();
						
					}
				}
			}
		});
	}

	protected void updateControls() {
		if (tree.getSelectionCount() == 0)
			return;
		TreeItem tiT = tree.getSelection()[0];
		File fileT = (File) tiT.getData();
		if (fileT.getName().endsWith(".zip")) {
			File fileRT = fileT.getParentFile();
			File fileSeries = fileRT.getParentFile();
			File fileTc = fileSeries.getParentFile();
			File fileVendor = fileTc.getParentFile();
			textType.setText(fileT.getName().replace(".zip", ""));
			textRTType.setText(fileRT.getName());
			textSeries.setText(fileSeries.getName() + " [" + fileTc.getName() + "]");
			textVendor.setText(fileVendor.getName());

			String description = null;
			try {
				String content = XmlMofidyUtil.getInstance().getChipContent(fileT);
				String[] arr = content.split("Description=");
				if (arr.length == 2) {
					description = arr[1].split("\r\n")[0];
				}
			} catch (Exception e1) {
				e1.printStackTrace();
			}
			if (description != null) {
				textDetail.setText(description.replace("\\n", "\n"));
			} else {
				textDetail.setText(EMPTY_STR);
			}
		}

	}

	public ArrayList<File> getInputList() {
		return inputList;
	}

	private void setSelection(File fileZip) {
		treeViewer.setSelection(new TreeSelection(new TreePath(new File[] {
				fileZip.getParentFile().getParentFile().getParentFile(), fileZip.getParentFile(), fileZip })));
	}
}
