/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package wizard;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Files;
import java.util.ArrayList;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import importOperation.ImportUtil;
public class NewProjectWizard extends Wizard implements INewWizard {

	private NewWizardPage newWizardPage;
	private String vendor;
	private String series;
	private String name;
	private String location;
	private String link;

	private String zipTemplatePath;
	private ArrayList<File> vendorlist = new ArrayList<File>();

	public NewProjectWizard() {
		setWindowTitle(Messages.NewProjectWizard_9); // $NON-NLS-1$
		setNeedsProgressMonitor(true);
		setForcePreviousAndNextButtons(true);
	}

	@Override
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		makeupVendorList();
	}

	public void makeupVendorList() {
		vendorlist.clear();
		String installPath = Platform.getInstallLocation().getURL().getFile();
		File fileWizardRoot = new File(installPath + "template" + File.separator + "wizard");
		if (fileWizardRoot.exists() && fileWizardRoot.isDirectory()) {
			File fileOthers = null;
			for (int i = 0; i < fileWizardRoot.listFiles().length; i++) {
				File afile = fileWizardRoot.listFiles()[i];
				if (afile.getName().equalsIgnoreCase("WCH")) {
					// set wch templates in the front
					vendorlist.add(0, afile);
				} else if (afile.getName().equalsIgnoreCase("Others")) {
					fileOthers = afile;
					continue;
				} else {
					vendorlist.add(afile);
				}
			}

			if (fileOthers != null) {
				// set others templates in the end
				vendorlist.add(fileOthers);
			}
		}
	}

	@Override
	public void addPages() {
		newWizardPage = new NewWizardPage(); // еĵһҳ
		newWizardPage.setErrorMessage(Messages.NewWizardPage_11);
		newWizardPage.setPageComplete(false);
		this.addPage(newWizardPage);
	}

	private void addFileRecursive(TreeItem parent) throws IOException {
		for (TreeItem child : parent.getItems()) {
			String strType = (String) child.getData("Type");
			if (strType.equals("File")) {
				String srcPath = (String) child.getData("SourcePath");
				String tgtPath = (String) parent.getData("TargetPath") + File.separator + child.getText();
				child.setData("TargetPath", tgtPath);
				if (!new File(tgtPath).exists()) {
					Files.copy(new File(srcPath).toPath(), new File(tgtPath).toPath());
				}
			} else if (strType.equals("Folder")) {
				String tgtPath = (String) parent.getData("TargetPath") + File.separator + child.getText();
				child.setData("TargetPath", tgtPath);
				File tgtFile = new File(tgtPath);
				if (!tgtFile.exists()) {
					tgtFile.mkdirs();
				}
				addFileRecursive(child);
			}
		}
	}

	@Override
	public boolean performFinish() {
		vendorlist.clear();
		String str = null;
		if (newWizardPage.getLocation() == null) {
			str = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString().replace("/", "\\") + File.separator
					+ newWizardPage.getName();
		} else {
			if (newWizardPage.getLocation().isEmpty()) {
				MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_1, Messages.NewProjectWizard_6);
				return false;
			}
			str = newWizardPage.getLocation() + File.separator + newWizardPage.getName();
		}

		IProject[] arrExistedPrjr = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		for (IProject prj : arrExistedPrjr) {
			if (prj.getName().equals(newWizardPage.getName())) {
				MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_7, Messages.NewProjectWizard_2);
				return false;
			}
		}

		if (new File(str).exists()) {
			MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_1, Messages.NewProjectWizard_2);
			newWizardPage.updateProjName(newWizardPage.getName());
			return false;
		}
//		if (ProjectUtil.isProjectHasExist(newWizardPage.getName())) {
//			MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_1, Messages.NewProjectWizard_2);
//
//			newWizardPage.updateProjName(newWizardPage.getName());
//			return false;
//		}
		if (newWizardPage.getSelectedTemplate() == null) { // $NON-NLS-1$
			MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_4, Messages.NewProjectWizard_5);
			return false;
		}
		if (newWizardPage.isSelfDefine()) {
			// Self-define template
			try {
				String templatePath = Platform.getInstallLocation().getURL().getFile() + "template";
				if (getPages().length == 1) {
					NewWizardPage newWizardPage = (NewWizardPage) getPages()[0];
					String projPath = templatePath + File.separator + "wizard" + File.separator
							+ newWizardPage.getVendor() + File.separator + newWizardPage.getSeries() + File.separator
							+ getPages()[0].getName();
					if (new File(projPath + ".zip").exists()) {
						if (!MessageDialog.openQuestion(getShell(), Messages.NewProjectWizard_10,
								Messages.NewProjectWizard_11)) {
							return false;
						}
					}
					File projDir = new File(projPath);
					if (!projDir.exists()) {
						projDir.mkdirs();
					}
					// copy standard .project
					File projDescriptionFile = new File(projPath + File.separator + ".project");
					if (projDescriptionFile.exists()) {
						projDescriptionFile.delete();
					}
					Files.copy(new File(templatePath + File.separator + ".project").toPath(),
							projDescriptionFile.toPath());
					// copy standard .cproject
					File projCProjFile = new File(projPath + File.separator + ".cproject");
					if (projCProjFile.exists()) {
						projCProjFile.delete();
					}

					Files.copy(new File(
							templatePath + File.separator + newWizardPage.getSelfDefineToolchain() + ".cproject")
									.toPath(),
							projCProjFile.toPath());
					// compress type folder to type.zip
					FileTool.compress(projPath, "zip");
				} else {
					for (IWizardPage page : getPages()) {
						if (page instanceof NewWizardPage2) {
							NewWizardPage2 page2 = (NewWizardPage2) page;
							TreeItem tiProj = page2.getTree().getItems()[0];
							String projPath = templatePath + File.separator + "wizard" + File.separator
									+ newWizardPage.getVendor() + File.separator + newWizardPage.getSeries()
									+ File.separator + tiProj.getText();
							if (new File(projPath + ".zip").exists()) {
								if (!MessageDialog.openQuestion(getShell(), Messages.NewProjectWizard_10,
										Messages.NewProjectWizard_11)) {
									return false;
								}
							}
							File projDir = new File(projPath);
							if (!projDir.exists()) {
								projDir.mkdirs();
							}
							tiProj.setData("TargetPath", projPath);
							addFileRecursive(tiProj);
							// compress type folder to type.zip
							FileTool.compress(projPath, "zip");
							break;
						}
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		zipTemplatePath = newWizardPage.getSelectedTemplate();
		if (!ProjectUtil.isValidTemplate(zipTemplatePath)) {
			MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_7, Messages.NewProjectWizard_8);
			return false;
		}
		vendor = newWizardPage.getVendor();
		series = newWizardPage.getSeries();
		name = newWizardPage.getName();
		location = newWizardPage.getLocation();
		link = newWizardPage.getDebuggerLink();

		// check Directory with specified name already exists
		if (ProjectUtil.isSpecificDirectoryExist(location + File.separator + name)) {
			boolean ret = MessageDialog.openQuestion(getShell(), Messages.NewProjectWizard_4,
					Messages.NewProjectWizard_3);
			if (!ret) {
				return false;
			}
		}
		showProgressDialog();

		if (newWizardPage.isSelfDefine()) {
			File createZipFile = new File(zipTemplatePath);
			File createSeriesFile = createZipFile.getParentFile();
			File createVendorFile = createSeriesFile.getParentFile();
			createZipFile.delete(); // delete temp zip
			createSeriesFile.delete(); // delete temp parent if empty
			createVendorFile.delete(); // delete temp parent if empty
		}
		// Get the active page.
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				IPerspectiveRegistry reg = PlatformUI.getWorkbench().getPerspectiveRegistry();

				IPerspectiveDescriptor finalPersp = reg.findPerspectiveWithId("intro.Perspectives");
				// Set the perspective.
				page.setPerspective(finalPersp);
			}
		}

		return true;
	}

	void createProject(boolean bCreateDir, String vendor, String series, String name, String location, String link,
			String zipTemplatePath, IProgressMonitor monitor) {
		try {
			ImportUtil.importProjectFromZip(bCreateDir, vendor, series, name, location, link, zipTemplatePath, monitor);
		} catch (Exception e) {
			e.printStackTrace();
			MessageDialog.openWarning(getShell(), Messages.NewProjectWizard_14, e.getMessage());
			return;
		}
	}

	private void showProgressDialog() {
		IRunnableWithProgress runnable = new IRunnableWithProgress() {

			@Override
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				monitor.beginTask(NLS.bind(Messages.NewProjectWizard_18, name), 10);
				createProject(newWizardPage.isbCreateDir(), vendor, series, name, location, link, zipTemplatePath,
						monitor);
				monitor.done();
			}
		};
		ProgressMonitorDialog progressMonitorDialog = new ProgressMonitorDialog(null);
		progressMonitorDialog.setCancelable(false);
		try {
			progressMonitorDialog.run(true, true, runnable);
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}

	public ArrayList<File> getVendorlist() {
		return vendorlist;
	}
}
