/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package wizard;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;

import menuAction.LoadAction;

public class NewSolutionWizard extends Wizard implements INewWizard {
	private static final String NEWLINE = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
	private IWorkbenchWindow fWindow;

	@Override
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		fWindow = workbench.getActiveWorkbenchWindow();
	}

	@Override
	public boolean performFinish() {
		NewSolutionDialog newSolutionDialog = new NewSolutionDialog(fWindow.getShell());
		if (Window.OK == newSolutionDialog.open()) {
			String savePath = newSolutionDialog.getFilePath();

			if (savePath != null) {
				File saveFile = new File(savePath);
				if (saveFile.exists()) {
					return false;
				}

				MessageDialog.openInformation(fWindow.getShell(), Messages.NewSolutionWizard_1,
						Messages.NewSolutionWizard_2);

				File workspaceFolder = saveFile.getParentFile();
				File metaFile = new File(workspaceFolder.getAbsolutePath() + File.separator + ".metadata");
				if (metaFile.exists()) {
					File backupFolder = new File(workspaceFolder.getAbsolutePath() + "_backup");
					if (!backupFolder.exists()) {
						backupFolder.mkdir();
					}

					try {
						copyToDir(workspaceFolder, backupFolder);
					} catch (IOException e) {
						e.printStackTrace();
					}

					// delete files except .plugins
					for (File f : metaFile.listFiles()) {
						if (!f.getName().equals("plugins")) {
							f.delete();
						}
					}

					// delete files except org.eclipse.core.resources
					File metaPlgFile = new File(workspaceFolder.getAbsolutePath() + File.separator + ".metadata"
							+ File.separator + ".plugins");
					if (metaPlgFile.exists()) {
						for (File f : metaPlgFile.listFiles()) {
							if (!f.getName().equals("org.eclipse.core.resources")) {
								f.delete();
							}
						}
					}
				}

				String buildToolsPath = newSolutionDialog.getBuildToolsPath();
				String armTcPath = newSolutionDialog.getArmTcPath();
				String rvTcPath = newSolutionDialog.getRvTcPath();
				String openocdPath = newSolutionDialog.getOpenocdPath();
				String openocdName = newSolutionDialog.getOpenocdName();

				FileWriter fileWriter = null;
				try {
					saveFile.createNewFile();
					fileWriter = new FileWriter(saveFile);
					fileWriter.write(savePath + ":" + NEWLINE); //$NON-NLS-1$
					fileWriter.write("-arm:" + armTcPath + NEWLINE);
					fileWriter.write("-build:" + buildToolsPath + NEWLINE);
					fileWriter.write("-openocd name:" + openocdName + NEWLINE);
					fileWriter.write("-openocd path:" + openocdPath + NEWLINE);
					fileWriter.write("-rv:" + rvTcPath + NEWLINE);
				} catch (IOException e) {
					e.printStackTrace();
				} finally {
					if (fileWriter != null) {
						try {
							fileWriter.close();
						} catch (IOException e) {
							e.printStackTrace();
						}
					}
				}

				try {
					LoadAction.changeSolution(saveFile);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		return false;
	}

	// public method for copy a file or folder to destination
	public static void copyToDir(File f1, File f2) throws IOException {
		if (!f2.isDirectory())
			return;
		if (f1.isDirectory()) {
			File[] subFiles = f1.listFiles();
			for (int i = 0; i < subFiles.length; i++) {
				String newFileName = f2.getPath() + File.separator + subFiles[i].getName();
				File newFile = new File(newFileName);
				if (subFiles[i].isDirectory()) {
					if (!newFile.exists()) {
						newFile.mkdir();
					} else {

					}
					copyToDir(subFiles[i], newFile);
				} else {
					copyToDir(subFiles[i], f2);
				}
			}
		} else {
			String newFileName = f2.getPath() + File.separator + f1.getName();
			File newFile = new File(newFileName);
			newFile.createNewFile();
			FileInputStream fis = new FileInputStream(f1);
			BufferedInputStream bis = new BufferedInputStream(fis, 8192);
			FileOutputStream fos = new FileOutputStream(newFileName);
			BufferedOutputStream bos = new BufferedOutputStream(fos, 8192);
			byte[] b = new byte[8192];
			int count = -1;
			while ((count = bis.read(b)) != -1) {
				bos.write(b, 0, count);
			}
			bis.close();
			bos.close();
			fis.close();
			fos.close();
		}
	}
}
