/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package wizard;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.wb.swt.ResourceManager;

import autoStart.AutoStartAction;
import convertProjectUtil.XmlMofidyUtil;
import mrsdebugger.Debugger;
import mrsdebugger.MCU;
import mrsdebugger.MRSComponent;

@SuppressWarnings("restriction")
public class NewWizardPage extends WizardPage {
	private static Image imgVendor = ResourceManager.getPluginImage("com.mounriver.project", "icon/vendor.png");
	private static Image imgSeries = ResourceManager.getPluginImage("com.mounriver.project", "icon/series.png");
	private static Image imgType = ResourceManager.getPluginImage("com.mounriver.project", "icon/type.png");

	private static final String EMPTY_STR = "";
	private static final char[] FORBIDDEN_CHAR = { '=', ',', '$', '#', '*', '&', '(', ')', '{', '}', '', '', '', '',
			'' };
	public static final String[] RTOSARR = { "NoneOS", "FreeRTOS", "Harmony LiteOS-M", "RT-Thread", "TencentOS" };

	private Text textProjName;
	private Text projectLocation;
	private Tree treeLeftExist;
	private Tree treeRightExist;
	
	private StyledText textIntro;
//	private Text textIntro;
	private Button btnSelfDefineMode;

	private boolean selfDefine = false;
	/** Is the next wizard page disabled? */
	private boolean fNextPageEnabled = false;
	private Composite container;

	private NewWizardPage2 page2;

	private boolean pageValid = true;
	private Text textFilter;
	private Button btnRV;
	private Button btnARM;
	private Combo comboSelfDefineToolchain;
	private Button createDirButton;
	private boolean bCreateDir = true;
	private Button browerButton;
	private Label labelLocation;
	private Combo comboRTOS;
	private Combo comboDebuggerLink;

//	private ArrayList<String> debuggerLinkList = new ArrayList<String>();

	public NewWizardPage() {
		super(Messages.NewWizardPage_1);
		setTitle(Messages.NewWizardPage_1);
		setDescription(Messages.NewWizardPage_2);
		setPageComplete(false);

		// add current built-in link items
//		debuggerLinkList.clear();
//		debuggerLinkList.add("WCH-Link");
//		debuggerLinkList.add("GDLink");
//		debuggerLinkList.add("JLink");
//		debuggerLinkList.add("Others");
	}

	@Override
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		container = new Composite(parent, SWT.FILL);
		setControl(container);
		GridLayout layout = new GridLayout(3, false);
		container.setLayout(layout);

		Label labelName = new Label(container, SWT.NONE);
		labelName.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
		labelName.setText(Messages.NewWizardPage_3);

		textProjName = new Text(container, SWT.BORDER);
		textProjName.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
		textProjName.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(ModifyEvent e) {
				String name = textProjName.getText();
				if (name.trim().isEmpty()) {
					setErrorMessage(Messages.NewWizardPage_11);
					pageValid = false;
					updateButtons();
					return;
				}
				// FORBIDDEN_CHAR = { '=', ',', '$', '#', '&', '(', ')', '{', '}' };
				for (char ch : FORBIDDEN_CHAR) {
					if (name.indexOf(ch) != -1) {
						setErrorMessage(NLS.bind(Messages.NewWizardPage_10, ch));
						pageValid = false;
						updateButtons();
						return;
					}
				}

				// INVALID_RESOURCE_CHARACTERS = new char[] {'\\', '/', ':', '*', '?', '"', '<',
				// '>', '|'};
				IStatus nameStatus = IDEWorkbenchPlugin.getPluginWorkspace().validateName(name, IResource.PROJECT);
				if (!nameStatus.isOK()) {
					setErrorMessage(nameStatus.getMessage());
					pageValid = false;
					updateButtons();
					return;
				}
				setErrorMessage(null);
				pageValid = true;
				updateButtons();
			}
		});

		Button useDefaultsButton = new Button(container, SWT.CHECK);
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1);
		gd.horizontalIndent = 5;
		gd.verticalIndent = 5;
		useDefaultsButton.setLayoutData(gd);
		useDefaultsButton.setText(Messages.NewWizardPage_4);
		useDefaultsButton.setSelection(true);
		useDefaultsButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				if (projectLocation != null) {
					if (useDefaultsButton.getSelection()) {
						createDirButton.setEnabled(false);
						createDirButton.setSelection(true);
						projectLocation.setText(getWorkSpacePath());
						projectLocation.setToolTipText(getWorkSpacePath());
						projectLocation.setEnabled(false);
						browerButton.setEnabled(false);
						labelLocation.setEnabled(false);
					} else {
						createDirButton.setEnabled(true);
						projectLocation.setText(EMPTY_STR);
						projectLocation.setEnabled(true);
						browerButton.setEnabled(true);
						labelLocation.setEnabled(true);
					}
					bCreateDir = createDirButton.getSelection();
				}
			}
		});

		createDirButton = new Button(container, SWT.CHECK);
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false, 2, 1);
		gd.horizontalIndent = 5;
		gd.verticalIndent = 5;
		createDirButton.setLayoutData(gd);
		createDirButton.setText(Messages.NewWizardPage_26);
		createDirButton.setSelection(true);
		createDirButton.setEnabled(false);
		createDirButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				bCreateDir = createDirButton.getSelection();
			}
		});

		labelLocation = new Label(container, SWT.NONE);
		labelLocation.setText(Messages.NewWizardPage_27);
		labelLocation.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
		labelLocation.setEnabled(false);

		projectLocation = new Text(container, SWT.READ_ONLY | SWT.BORDER);
		projectLocation.setText(getWorkSpacePath());
		projectLocation.setToolTipText(getWorkSpacePath());
		projectLocation.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));
		projectLocation.setEnabled(false);

		browerButton = new Button(container, SWT.PUSH);
		browerButton.setText(Messages.NewWizardPage_6);
		GridData gData = new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1);
		browerButton.setLayoutData(gData);
		browerButton.setEnabled(false);
		browerButton.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				DirectoryDialog directoryDialog = new DirectoryDialog(container.getShell());
				directoryDialog.setText(Messages.NewWizardPage_7);
				directoryDialog.setMessage(Messages.NewWizardPage_8);
				String path = directoryDialog.open();
				if (path != null) {
					projectLocation.setText(path);
					projectLocation.setToolTipText(path);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
			}
		});

		Composite groupSelfDefine = new Composite(container, SWT.NONE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		groupSelfDefine.setLayoutData(gd);
		groupSelfDefine.setLayout(new GridLayout(3, false));

		btnSelfDefineMode = new Button(groupSelfDefine, SWT.CHECK);
		btnSelfDefineMode.setText(Messages.NewWizardPage_20);
		btnSelfDefineMode.setSelection(selfDefine);
		btnSelfDefineMode.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));

		Label labelSelfDefineToolchain = new Label(groupSelfDefine, SWT.NONE);
		gd = new GridData();
		gd.horizontalIndent = 10;
		labelSelfDefineToolchain.setLayoutData(gd);
		labelSelfDefineToolchain.setText(Messages.NewWizardPage_25);

		comboSelfDefineToolchain = new Combo(groupSelfDefine, SWT.DROP_DOWN | SWT.READ_ONLY);
		comboSelfDefineToolchain.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));
		comboSelfDefineToolchain.setItems(new String[] { "RISC-V Cross GCC", "ARM Cross GCC" });
//		comboSelfDefineToolchain.setItems(new String[] { "RISC-V Cross GCC" });
		comboSelfDefineToolchain.select(0);
		comboSelfDefineToolchain.setEnabled(false);

		Group groupTemplate = new Group(container, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 3;
//		gd.heightHint = 500;
//		gd.widthHint = 1000;
		groupTemplate.setLayoutData(gd);
		groupTemplate.setLayout(new GridLayout(2, true));

		Composite compTc = new Composite(groupTemplate, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		compTc.setLayoutData(gd);
		compTc.setLayout(new GridLayout(2, false));

		btnRV = new Button(compTc, SWT.CHECK);
		btnRV.setText(Messages.NewWizardPage_0);

		btnARM = new Button(compTc, SWT.CHECK);
		btnARM.setText(Messages.NewWizardPage_24);

		btnRV.setSelection(true);
		btnARM.setSelection(true);

		Composite compRT = new Composite(groupTemplate, SWT.NONE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		compRT.setLayoutData(gd);
		compRT.setLayout(new GridLayout(2, false));

		Label labelRTOS = new Label(compRT, SWT.NONE);
		labelRTOS.setText(Messages.NewWizardPage_28);

		comboRTOS = new Combo(compRT, SWT.DROP_DOWN | SWT.READ_ONLY);
		comboRTOS.setItems(RTOSARR);
		comboRTOS.select(0);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		comboRTOS.setLayoutData(gd);
		comboRTOS.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				for (TreeItem treeItem : treeLeftExist.getItems()) {
					treeItem.dispose();
				}
				for (TreeItem treeItem : treeRightExist.getItems()) {
					treeItem.dispose();
				}
				redraw(btnRV.getSelection(), btnARM.getSelection(), comboRTOS.getSelectionIndex(),
						textFilter.getText());
				if (treeRightExist.getSelectionCount() > 0) {
					updateProjName(treeRightExist.getSelection()[0].getText());
					updateStatus((File) treeRightExist.getSelection()[0].getData());
				}
			}
		});

		Composite compFilter = new Composite(groupTemplate, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
//		gd.minimumHeight = 100;
		compFilter.setLayoutData(gd);
		compFilter.setLayout(new GridLayout(2, false));

		Label labelFilter = new Label(compFilter, SWT.BOLD);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		labelFilter.setLayoutData(gd);
		labelFilter.setText(Messages.NewWizardPage_23);

		textFilter = new Text(compFilter, SWT.BORDER);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		gd.widthHint = 190;
		textFilter.setLayoutData(gd);

		treeLeftExist = new Tree(groupTemplate, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 140;
		gd.minimumHeight = 50;
		treeLeftExist.setLayoutData(gd);

		treeRightExist = new Tree(groupTemplate, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 140;
		gd.minimumHeight = 50;
		treeRightExist.setLayoutData(gd);

		Composite compDebugger = new Composite(container, SWT.NONE);
		gd = new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1);
		gd.horizontalSpan = 3;
//		gd.minimumHeight = 40;
		compDebugger.setLayoutData(gd);
		compDebugger.setLayout(new GridLayout(2, false));

		Label labelLink = new Label(compDebugger, SWT.NONE);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		labelLink.setLayoutData(gd);
		labelLink.setText(Messages.NewWizardPage_29);

		comboDebuggerLink = new Combo(compDebugger, SWT.DROP_DOWN | SWT.READ_ONLY);
//		comboDebuggerLink.setItems(debuggerLinkList.toArray(new String[debuggerLinkList.size()]));
//		comboDebuggerLink.select(0);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		comboDebuggerLink.setLayoutData(gd);

		Label labelIntro = new Label(container, SWT.BOLD);
		gd = new GridData(GridData.BEGINNING);
		gd.horizontalSpan = 3;
		labelIntro.setLayoutData(gd);
		labelIntro.setText(Messages.NewWizardPage_17);
		

		textIntro = new StyledText(container, SWT.BORDER | SWT.WRAP | SWT.READ_ONLY | SWT.MULTI | SWT.V_SCROLL);
		textIntro.setEditable(false);
		textIntro.setBackground(new Color(Display.getCurrent(), 240, 240, 240));
//		gd = new GridData(SWT.FILL, SWT.FILL, false, true, 1, 1);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 3;
		gd.widthHint = 677;
		gd.heightHint = 155;
		textIntro.setLayoutData(gd);

		redraw(true, true, 0, EMPTY_STR);

		if (treeRightExist.getItemCount() > 0) {
			String name = treeRightExist.getItem(0).getText();
			updateProjName(name == null ? EMPTY_STR : name);
		}

		addListeners();
	}

	/**
	 * redraw tree
	 * 
	 * @param showRV
	 * @param showARM
	 * @param filter
	 */
	private void redraw(boolean showRV, boolean showARM, int index, String filter) {
		addItemsToLeftTree(showRV, showARM, index, filter);
		initSelection(showRV, showARM, index, filter);
		textFilter.setFocus();
	}

	@Override
	public boolean canFlipToNextPage() {
		return fNextPageEnabled;
	}

	public String getName() {
		return textProjName.getText();
	}

	public String getWorkSpacePath() {
		return ResourcesPlugin.getWorkspace().getRoot().getLocation().toString().replace("/", "\\");
	}

	public String getLocation() {
		String loc = projectLocation.getText();
		if (loc.equals(getWorkSpacePath())) {
			return null;
		}
		return loc.replace("\\", "/");
	}

	private void addItemsToLeftTree(boolean showRV, boolean showARM, int index, String filter) {
		NewProjectWizard wizard = (NewProjectWizard) getWizard();
		wizard.makeupVendorList();
		for (File fileV : wizard.getVendorlist()) {
			if (fileV.isDirectory()) {
				String vendor = fileV.getName();
				TreeItem tiVExist = null;
				List<File> sortedTcList = Arrays.asList(fileV.listFiles());
				Collections.sort(sortedTcList, new Comparator<File>() {

					@Override
					public int compare(File arg0, File arg1) {
						if (arg0.getName().equals("RISC-V")) {
							// set rv chips
							return -1;
						}

						if (arg1.getName().equals("ARM")) {
							return 1;
						}
						return arg0.getName().compareTo(arg1.getName());
					}
				});

				for (File fileTc : sortedTcList) {
					if ((fileTc.getName().equals("RISC-V") && !showRV)
							|| (fileTc.getName().equals("ARM") && !showARM)) {
						continue;
					}

					if (fileTc.isDirectory()) {
						List<File> sortedSeriesList = Arrays.asList(fileTc.listFiles());
						Collections.sort(sortedSeriesList, new Comparator<File>() {

							@Override
							public int compare(File arg0, File arg1) {
								return arg0.getName().compareTo(arg1.getName());
							}

						});

						for (File fileS : sortedSeriesList) {
							if (fileS.isDirectory()) {
								boolean bFound = false;
								for (File fileRT : fileS.listFiles()) {
									if (fileRT.isDirectory() && RTOSARR[index].equals(fileRT.getName())) {
										for (File fileT : fileRT.listFiles()) {
											if (fileT.getName().endsWith(".zip") && (filter.isEmpty()
													|| fileT.getName().toUpperCase().contains(filter.toUpperCase()))) {
												bFound = true;
												break;
											}
										}
									}
								}

								if (bFound) {
									if (tiVExist == null) {
										tiVExist = new TreeItem(treeLeftExist, SWT.NONE);
									}
									tiVExist.setData(fileV);
									tiVExist.setImage(imgVendor);
									tiVExist.setText(vendor);

									TreeItem tiSExist = new TreeItem(tiVExist, SWT.NONE);
									String series = fileS.getName();
									tiSExist.setData(fileS);
									tiSExist.setImage(imgSeries);
									tiSExist.setExpanded(true);
									tiSExist.setText(series + " [" + fileTc.getName() + "]");
								}
							}
						}
					}
					if (tiVExist != null) {
						tiVExist.setExpanded(true);
					}
				}
			}
		}
	}

	private void initSelection(boolean showRV, boolean showARM, int index, String filter) {
		if (treeLeftExist.getItemCount() > 0) {
			TreeItem tiV0Exist = treeLeftExist.getItem(0);
			if (tiV0Exist.getItemCount() > 0) {
				TreeItem tiS0Exist = tiV0Exist.getItem(0);
				File fileS0Exist = (File) tiS0Exist.getData();
				if (!fileS0Exist.isDirectory())
					return;
				treeLeftExist.select(tiS0Exist);

				boolean hasRTOS = false;
				File file = (File) tiS0Exist.getData();
				if (file.isDirectory()) {
					for (File fileRT : file.listFiles()) {
						if (fileRT.getName().equals("RTOS") && fileRT.isDirectory()) {
							for (File fileType : fileRT.listFiles()) {
								if (fileType.isFile() && fileType.getName().endsWith(".zip")) {
									hasRTOS = true;
									break;
								}
							}
							if (hasRTOS) {
								break;
							}
						}
					}
				}
//				btnRadioRTOS.setEnabled(hasRTOS);

				for (int i = 0; i < fileS0Exist.listFiles().length; i++) {
					File fileRT = fileS0Exist.listFiles()[i];
					if (fileRT.isDirectory() && RTOSARR[index].equals(fileRT.getName())) {
						List<File> sortedTypeList = Arrays.asList(fileRT.listFiles());
						Collections.sort(sortedTypeList, new Comparator<File>() {
							@Override
							public int compare(File arg0, File arg1) {
								return arg0.getName().compareTo(arg1.getName());
							}

						});

						for (File fileTExist : sortedTypeList) {
							if (!filter.isEmpty()
									&& !fileTExist.getName().toUpperCase().contains(filter.toUpperCase())) {
								continue;
							}
							int suffixPos = fileTExist.getName().indexOf(".zip");
							if (suffixPos != -1) {
								TreeItem tiTExist = new TreeItem(treeRightExist, SWT.NONE);
								tiTExist.setImage(imgType);
								String name = fileTExist.getName().substring(0, suffixPos);
								tiTExist.setText(name);
								tiTExist.setData(fileTExist);
							}
						}
					}
				}
				if (treeRightExist.getItemCount() > 0) {
					treeRightExist.setSelection(treeRightExist.getItem(0));
					updateStatus((File) treeRightExist.getItem(0).getData());
				} else {
					textIntro.setText(EMPTY_STR);
				}
			}
		}
		treeLeftExist.setEnabled(!btnSelfDefineMode.getSelection());
		treeRightExist.setEnabled(!btnSelfDefineMode.getSelection());
	}

	private void addListeners() {
		btnRV.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				for (TreeItem treeItem : treeLeftExist.getItems()) {
					treeItem.dispose();
				}
				for (TreeItem treeItem : treeRightExist.getItems()) {
					treeItem.dispose();
				}

				redraw(btnRV.getSelection(), btnARM.getSelection(), comboRTOS.getSelectionIndex(),
						textFilter.getText());
				if (treeRightExist.getSelectionCount() > 0) {
					updateProjName(treeRightExist.getSelection()[0].getText());
					updateStatus((File) treeRightExist.getSelection()[0].getData());
				}
			}
		});

		btnARM.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				for (TreeItem treeItem : treeLeftExist.getItems()) {
					treeItem.dispose();
				}
				for (TreeItem treeItem : treeRightExist.getItems()) {
					treeItem.dispose();
				}
				redraw(btnRV.getSelection(), btnARM.getSelection(), comboRTOS.getSelectionIndex(),
						textFilter.getText());
				if (treeRightExist.getSelectionCount() > 0) {
					updateProjName(treeRightExist.getSelection()[0].getText());
					updateStatus((File) treeRightExist.getSelection()[0].getData());
				}
			}
		});

		textFilter.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(ModifyEvent e) {
				for (TreeItem treeItem : treeLeftExist.getItems()) {
					treeItem.dispose();
				}
				for (TreeItem treeItem : treeRightExist.getItems()) {
					treeItem.dispose();
				}
				redraw(btnRV.getSelection(), btnARM.getSelection(), comboRTOS.getSelectionIndex(),
						textFilter.getText());
				if (treeRightExist.getSelectionCount() > 0) {
					updateProjName(treeRightExist.getSelection()[0].getText());
					updateStatus((File) treeRightExist.getSelection()[0].getData());
				}
			}
		});

		btnSelfDefineMode.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (btnSelfDefineMode.getSelection()) {
					comboSelfDefineToolchain.setEnabled(true);
					textFilter.setEnabled(false);
					treeLeftExist.setEnabled(false);
					treeRightExist.setEnabled(false);
					btnRV.setEnabled(false);
					btnARM.setEnabled(false);
					comboRTOS.setEnabled(false);
					comboDebuggerLink.setEnabled(false);

					updateProjName(EMPTY_STR);
					selfDefine = true;
					updateStatus(null);
				} else {
					comboSelfDefineToolchain.setEnabled(false);
					textFilter.setEnabled(true);
					treeLeftExist.setEnabled(true);
					treeRightExist.setEnabled(true);
					btnRV.setEnabled(true);
					btnARM.setEnabled(true);
					comboRTOS.setEnabled(true);
					comboDebuggerLink.setEnabled(true);

					selfDefine = false;
					if (treeRightExist.getSelectionCount() > 0) {
						updateProjName(treeRightExist.getSelection()[0].getText());
						updateStatus((File) treeRightExist.getSelection()[0].getData());
					} else {
						textIntro.setText(EMPTY_STR);
					}
				}
			}
		});

		treeLeftExist.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				if (e.item instanceof TreeItem) {
					TreeItem ti = (TreeItem) e.item;
					if (ti.getItemCount() > 0 && ti.getExpanded()) {
						ti = ti.getItem(0);
					}
					treeLeftExist.select(ti);

					File file = (File) ti.getData();
					redrawRightTree(file);
				}
			}
		});

		treeRightExist.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeItem tiT = treeRightExist.getSelection()[0];
				File fileT = (File) tiT.getData();
				updateProjName(tiT.getText());
				updateStatus(fileT);

			}
		});

		comboDebuggerLink.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setErrorMessage(null);
				pageValid = true;
				updateButtons();
			}
		});

		textIntro.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				if ((e.button == 1) && (Platform.getOS().equals("win32"))) {
					try {
						Point point = new Point(e.x, e.y);
						int n = textIntro.getOffsetAtLocation(point);
						if (n == -1) {
							return;
						}
						StyleRange style = textIntro.getStyleRangeAtOffset(n);
						if (style != null && style.underlineStyle == 4) {
							String strCommand = "Explorer \"" + textIntro.getData().toString() + "\"";
							Runtime.getRuntime().exec(strCommand.split("\\s+"), new String[0]);
						}
					} catch (Exception e1) {
						// TODO: handle exception
					}
				}
			}
		});
	}

	protected void redrawRightTree(File series) {
		treeRightExist.removeAll();
		if (series.isDirectory()) {
			for (int i = 0; i < series.listFiles().length; i++) {
				File fileRT = series.listFiles()[i];
				int index = comboRTOS.getSelectionIndex();
				if (fileRT.isDirectory() && RTOSARR[index].equals(fileRT.getName())) {
					List<File> sortedTypeList = Arrays.asList(fileRT.listFiles());
					Collections.sort(sortedTypeList, new Comparator<File>() {

						@Override
						public int compare(File arg0, File arg1) {
							return arg0.getName().compareTo(arg1.getName());
						}

					});

					for (File fileT : sortedTypeList) {
						int suffixPos = fileT.getName().indexOf(".zip");
						if (suffixPos != -1
								&& fileT.getName().toUpperCase().contains(textFilter.getText().toUpperCase())) {
//						isFound = true;
							TreeItem tiT = new TreeItem(treeRightExist, SWT.NONE);
							String name = fileT.getName().substring(0, suffixPos);
							tiT.setText(name);
							tiT.setData(fileT);
							tiT.setImage(imgType);
						}
					}
				}
			}
		}

		if (treeRightExist.getItemCount() > 0) {
			treeRightExist.setSelection(treeRightExist.getItem(0));
			updateProjName(treeRightExist.getItem(0).getText());
			updateStatus((File) treeRightExist.getItem(0).getData());
		} else {
			textIntro.setText(EMPTY_STR);
		}

	}

	private void updateStatus(File fileT) {
		if (fileT != null) {
			String description = null;
			try {
				String content = XmlMofidyUtil.getInstance().getChipContent(fileT);
				String[] arr = content.split("Description=");
				if (arr.length == 2) {
					description = arr[1].split("\r\n")[0];
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
			if (description != null) {
				description = description.replace("\\n", "\n");
				textIntro.setText(description);
				// http://www.wch.cn/products/CH32V103.html?
				int indexStart = description.indexOf("http:");
				if (indexStart != -1) {
					int indexEnd = description.indexOf("\n", indexStart);
					String strUrl = description.substring(indexStart, indexEnd + 1);
					// add url style in description
					StyleRange styleRange = new StyleRange();
					styleRange.start = indexStart;
					styleRange.length = strUrl.length();
					styleRange.underline = true;
					styleRange.font = Display.getCurrent().getSystemFont();
					styleRange.underlineStyle = SWT.UNDERLINE_LINK;
					textIntro.setStyleRange(styleRange);
					textIntro.setData(strUrl);
					textIntro.update();
				}
			} else {
				textIntro.setText(EMPTY_STR);
			}

			ZipFile zip = null;
			try {
				zip = new ZipFile(fileT);
				StringBuffer stringBuffer = new StringBuffer();
				// ѹļ
				for (Enumeration<?> entries = zip.entries(); entries.hasMoreElements();) {
					ZipEntry entry = (ZipEntry) entries.nextElement();
					String zipEntryName = entry.getName();
					if (zipEntryName.equals(".template")) {
						InputStream is = zip.getInputStream(entry);
						byte buf[] = new byte[1024];
						int count = -1;
						while ((count = is.read(buf)) > -1) {
							String str = new String(buf, 0, count);
							stringBuffer.append(str);
						}
					}

				}
				String strContent = stringBuffer.toString();

				int index = strContent.indexOf("Vendor=");
				if (index != -1) {
					String zipVendor = strContent.substring(strContent.indexOf("Vendor=") + "Vendor=".length(),
							strContent.indexOf("\n", strContent.indexOf("Vendor=") + "Vendor=".length()));
					if (zipVendor.endsWith("\r")) {
						zipVendor = zipVendor.substring(0, zipVendor.length() - 1);
					}

					switch (zipVendor) {
					case "WCH":
						comboDebuggerLink.setItems(new String[] { "WCH-Link" });
						break;
					case "GD":
						comboDebuggerLink.setItems(new String[] { "GDLink", "JLink" });
						break;
					case "Others":
						comboDebuggerLink.setItems(new String[] { "Others" });
						break;
					default:
						ArrayList<MRSComponent> componentList = AutoStartAction.getMrsComponentList();
						MRSComponent mrsComponent = null;
						for (MRSComponent mrsComponent2 : componentList) {
							if (mrsComponent2.getVendor().equals(zipVendor)) {
								mrsComponent = mrsComponent2;
								break;
							}
						}
						if (mrsComponent != null) {
							ArrayList<MCU> mculist = mrsComponent.getMcuList();

							String[] strArr = null;
							for (int i = 0; i < mculist.size(); i++) {
								MCU mcu = mculist.get(i);
								if (mcu.getName().equals(fileT.getName().substring(0, fileT.getName().length() - 4))) {
									ArrayList<Debugger> debuggerList = mcu.getDebuggerList();
									Collections.sort(debuggerList, new Comparator<Debugger>() {

										@Override
										public int compare(Debugger arg0, Debugger arg1) {
											if (arg0.getName().equals("WCH-Link")) {
												return -1;
											}

											if (arg1.getName().equals("WCH-Link")) {
												return 1;
											}
											return arg0.getName().compareTo(arg1.getName());
										}
									});

									strArr = new String[debuggerList.size()];
									for (int j = 0; j < debuggerList.size(); j++) {
										strArr[j] = debuggerList.get(j).getName();
									}
									break;
								}
							}
							if (strArr != null) {
								comboDebuggerLink.setItems(strArr);
							}
						} else {
							comboDebuggerLink.setItems(new String[] {});
						}
					}
				} else {
					comboDebuggerLink.setItems(new String[] { "WCH-Link", "GDLink", "JLink", "Others" });
				}

				index = strContent.indexOf("Link=");
				if (index != -1) {
					int indexLink = -1;
					String zipLink = null;
					if (strContent.contains("\r\n")) {
						zipLink = strContent.substring(strContent.indexOf("Link=") + "Link=".length(),
								strContent.indexOf("\r\n", strContent.indexOf("Link=") + "Link=".length()));
					} else if (strContent.contains("\n")) {
						zipLink = strContent.substring(strContent.indexOf("Link=") + "Link=".length(),
								strContent.indexOf("\n", strContent.indexOf("Link=") + "Link=".length()));
					}
					if (zipLink != null) {
						List<String> list = new ArrayList<>(Arrays.asList(comboDebuggerLink.getItems()));
						indexLink = list.indexOf(zipLink);
					}
					if (indexLink == -1) {
						setErrorMessage(NLS.bind(Messages.NewWizardPage_30, zipLink));
						pageValid = false;
						updateButtons();
					}
					comboDebuggerLink.select(indexLink);
				} else {
					comboDebuggerLink.select(0);
				}

			} catch (Exception e1) {
				e1.printStackTrace();
			} finally {
				try {
					if (zip != null)
						zip.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}

		} else {
			textIntro.setText(Messages.NewWizardPage_22);
		}
		updateButtons();
	}

	private void updateButtons() {
		// this version not show next page.
		fNextPageEnabled = false;
		setPageComplete(pageValid);
	}

	public void updateProjName(String name) {
		textProjName.setText(name);
	}

	@Override
	public IWizardPage getNextPage() {
		if (page2 == null) {
			page2 = new NewWizardPage2(getName());
			NewProjectWizard wizard = (NewProjectWizard) getWizard();
			wizard.addPage(page2);
		} else {
			page2.modifyProjName(getName());
		}
		setPageComplete(true);
		return page2;
	}

	public String getVendor() {
		if (selfDefine) {
			return EMPTY_STR;
		} else {
			return treeLeftExist.getSelection()[0].getParentItem().getText();
		}
	}

	public String getDebuggerLink() {
		return comboDebuggerLink.getText();
	}

	private String getTc() {
		if (selfDefine) {
			return EMPTY_STR;
		} else {
			String text = treeLeftExist.getSelection()[0].getText();
			String regex = "(.*)\\[(.*)\\]";
			Matcher matcher = Pattern.compile(regex).matcher(text);
			if (Pattern.matches(regex, text) && matcher.find()) {
				return matcher.group(2);
			}
			return EMPTY_STR;
		}
	}

	public String getSeries() {
		if (selfDefine) {
			return EMPTY_STR;
		} else {
			String text = treeLeftExist.getSelection()[0].getText();
			String regex = "(.*)\\[(.*)\\]";
			Matcher matcher = Pattern.compile(regex).matcher(text);
			if (Pattern.matches(regex, text) && matcher.find()) {
				return matcher.group(1);
			}
			return EMPTY_STR;
		}
	}

	public String getSelectedTemplate() {
		if (!selfDefine) {
			TreeItem[] tiSelect = treeRightExist.getSelection();
			if (tiSelect.length > 0) {
				File fSelect = (File) (tiSelect[0].getData());
				return fSelect.getAbsolutePath();
			}
		} else {
			if (page2 == null) {
				String str = Platform.getInstallLocation().getURL().getFile() + "template" + File.separator + "wizard"
						+ File.separator + getVendor() + File.separator + getTc() + File.separator + getSeries()
						+ File.separator + getName() + ".zip";
				File file = new File(str);
				return file.getAbsolutePath();
			}
			return (String) page2.getTree().getItem(0).getData("TargetPath") + ".zip";
		}
		return null;
	}

	public boolean isSelfDefine() {
		return selfDefine;
	}

	public String getSelfDefineToolchain() {
		if (comboSelfDefineToolchain.getSelectionIndex() == 0) {
			return "RISC-V";
		}
		if (comboSelfDefineToolchain.getSelectionIndex() == 1) {
			return "ARM";
		}
		return EMPTY_STR;
	}

	public boolean isbCreateDir() {
		return bCreateDir;
	}
}
