/*******************************************************************************
 * Copyright (c) 2022, MounRiver Studio
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    MounRiver Studio  - initial API and implementation
 *******************************************************************************/
package wizard;

import java.io.File;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.wb.swt.ResourceManager;

public class NewWizardPage2 extends WizardPage {
	private static Image imgProject = ResourceManager.getPluginImage("com.mounriver.project", "icon/project.png"); //$NON-NLS-1$ //$NON-NLS-2$
	private static Image imgFolder = ResourceManager.getPluginImage("com.mounriver.project", "icon/folder.png"); //$NON-NLS-1$ //$NON-NLS-2$
	private static Image imgFile = ResourceManager.getPluginImage("com.mounriver.project", "icon/file.png"); //$NON-NLS-1$ //$NON-NLS-2$

	private Tree tree;
	private TreeItem tiProj;
	private String projName;
	private Button btnAddFolder;
	private Button btnAddFile;
	private Button btnCreateFolder;
	private Button btnDelete;

	protected NewWizardPage2(String projName) {
		super(projName);
		this.projName = projName;
		setTitle(Messages.NewWizardPage2_0);
		setDescription(Messages.NewWizardPage2_1);
		setPageComplete(true);
	}

	@Override
	public void createControl(Composite parent) {
		Composite userComp = new Composite(parent, SWT.FILL);
		setControl(userComp);
		userComp.setLayout(new GridLayout(4, false));

		tree = new Tree(userComp, SWT.SINGLE | SWT.BORDER);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 4;
		tree.setLayoutData(gd);

		btnAddFolder = new Button(userComp, SWT.PUSH);
		btnAddFolder.setText(Messages.NewWizardPage2_2);
		gd = new GridData();
		btnAddFolder.setLayoutData(gd);

		btnAddFile = new Button(userComp, SWT.PUSH);
		btnAddFile.setText(Messages.NewWizardPage2_3);
		gd = new GridData();
		btnAddFile.setLayoutData(gd);

		btnCreateFolder = new Button(userComp, SWT.PUSH);
		btnCreateFolder.setText(Messages.NewWizardPage2_4);
		gd = new GridData();
		btnCreateFolder.setLayoutData(gd);

		btnDelete = new Button(userComp, SWT.PUSH);
		btnDelete.setText(Messages.NewWizardPage2_5);
		gd = new GridData();
		btnDelete.setLayoutData(gd);

		initSelection();
		addListeners();
	}

	private void addListeners() {
		tree.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeItem item = (TreeItem) e.item;
				String type = (String) item.getData("Type"); //$NON-NLS-1$
				if (type.equals("Project")) { //$NON-NLS-1$
					btnDelete.setEnabled(false);
					return;
				}
				if (type.equals("File")) { //$NON-NLS-1$
					btnAddFolder.setEnabled(false);
					btnAddFile.setEnabled(false);
					btnCreateFolder.setEnabled(false);
					return;
				}
				btnAddFolder.setEnabled(true);
				btnAddFile.setEnabled(true);
				btnCreateFolder.setEnabled(true);
				btnDelete.setEnabled(true);
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
			}
		});

		btnAddFolder.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (tree.getSelectionCount() > 0) {
					TreeItem tiSelect = tree.getSelection()[0];
					DirectoryDialog directoryDialog = new DirectoryDialog(getShell(), SWT.OPEN | SWT.SHEET | SWT.MULTI);
					String path = directoryDialog.open();
					if (path == null)
						return;
					File file = new File(path);
					if (file.exists() && file.isDirectory()) {
						TreeItem ti = new TreeItem(tiSelect, SWT.NONE);
						ti.setImage(imgFolder);
						ti.setData("Type", "Folder"); //$NON-NLS-1$ //$NON-NLS-2$
						ti.setData("SourcePath", file.getAbsolutePath()); //$NON-NLS-1$
						ti.setText(file.getName());
						ti.getParentItem().setExpanded(true);
						addChildNodesInFolder(ti);
					}
				}
			}
		});

		btnAddFile.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (tree.getSelectionCount() > 0) {
					TreeItem tiSelect = tree.getSelection()[0];
					TreeItem[] existedItems = tiSelect.getItems();
					FileDialog fileDialog = new FileDialog(getShell(), SWT.OPEN | SWT.MULTI | SWT.SHEET);
					String path = fileDialog.open();
					if (path == null)
						return;
					String filterPath = fileDialog.getFilterPath();
					String[] playNames = fileDialog.getFileNames();
					for (String p : playNames) {
						if (existedItems.length > 0) {
							TreeItem duplicateItem = null;
							for (TreeItem existedItem : existedItems) {
								if (existedItem.getText().equals(p) && existedItem.getData("Type").equals("File")) {
									duplicateItem = existedItem;
									break;
								}
							}
							if (duplicateItem != null) {
								boolean result = MessageDialog.openQuestion(getShell(), "Question", "File " + p
										+ " already exists. Whether to overlap? (Otherwise skip this file)");
								if (result) {
									duplicateItem.dispose();
								} else {
									continue;
								}
							}
						}
						String fullName = filterPath + File.separator + p;
						File file = new File(fullName);
						if (file.exists() && file.isFile()) {
							TreeItem ti = new TreeItem(tiSelect, SWT.NONE);
							ti.setImage(imgFile);
							ti.setData("Type", "File"); //$NON-NLS-1$ //$NON-NLS-2$
							ti.setData("SourcePath", file.getAbsolutePath()); //$NON-NLS-1$
							ti.setText(file.getName());
							ti.getParentItem().setExpanded(true);
						}
					}
				}
			}
		});

		btnCreateFolder.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (tree.getSelectionCount() > 0) {
					CreateFolderDialog createFolderDialog = new CreateFolderDialog(getShell());
					if (createFolderDialog.open() == Window.OK) {
						TreeItem tiSelect = tree.getSelection()[0];
						TreeItem ti = new TreeItem(tiSelect, SWT.NONE);
						ti.setText(createFolderDialog.getFolderName());
						ti.setData("Type", "Folder"); //$NON-NLS-1$ //$NON-NLS-2$
						ti.setData("SourcePath", null); //$NON-NLS-1$
						ti.setImage(imgFolder);
						ti.getParentItem().setExpanded(true);
					}
				}
			}
		});

		btnDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (tree.getSelectionCount() > 0) {
					TreeItem item = tree.getSelection()[0];
					item.removeAll();
					item.dispose();
				}
			}
		});
	}

	private void initSelection() {
		tiProj = new TreeItem(tree, SWT.NONE);
		tiProj.setText(projName);
		tiProj.setData("Type", "Project"); //$NON-NLS-1$ //$NON-NLS-2$
		tiProj.setData("SourcePath", null); //$NON-NLS-1$
		tiProj.setImage(imgProject);

		TreeItem tiP1 = new TreeItem(tiProj, SWT.NONE);
		tiP1.setText(".project");
		tiP1.setData("Type", "File"); //$NON-NLS-1$ //$NON-NLS-2$
		tiP1.setData("SourcePath", //$NON-NLS-1$
				Platform.getInstallLocation().getURL().getFile() + "template" + File.separator + ".project");
		tiP1.setImage(imgFile);

		TreeItem tiP2 = new TreeItem(tiProj, SWT.NONE);
		tiP2.setText(".cproject");
		tiP2.setData("Type", "File"); //$NON-NLS-1$ //$NON-NLS-2$
		tiP2.setData("SourcePath", //$NON-NLS-1$
				Platform.getInstallLocation().getURL().getFile() + "template" + File.separator + ".cproject");
		tiP2.setImage(imgFile);

		tree.select(tiProj);
		tiProj.setExpanded(true);
	}

	private void addChildNodesInFolder(TreeItem parent) {
		String parentFilePath = (String) parent.getData("SourcePath"); //$NON-NLS-1$
		File[] children = new File(parentFilePath).listFiles();
		for (File child : children) {
			if (child.isDirectory()) {
				TreeItem tiFolder = new TreeItem(parent, SWT.NONE);
				tiFolder.setImage(imgFolder);
				tiFolder.setData("Type", "Folder"); //$NON-NLS-1$ //$NON-NLS-2$
				tiFolder.setData("SourcePath", child.getAbsolutePath()); //$NON-NLS-1$
				tiFolder.setText(child.getName());
				addChildNodesInFolder(tiFolder);
			} else {
				TreeItem tiFile = new TreeItem(parent, SWT.NONE);
				tiFile.setImage(imgFile);
				tiFile.setData("Type", "File"); //$NON-NLS-1$ //$NON-NLS-2$
				tiFile.setData("SourcePath", child.getAbsolutePath()); //$NON-NLS-1$
				tiFile.setText(child.getName());
			}
		}
	}

	@Override
	public boolean canFlipToNextPage() {
		return false;
	}

	public Tree getTree() {
		return tree;
	}

	public void modifyProjName(String name) {
		tiProj.setText(name);
	}
}
